package uk.ac.warwick.util.web.useragent;

import org.junit.Test;

import static org.junit.Assert.*;
import static uk.ac.warwick.util.web.useragent.UserAgentParser.INSTANCE;

public class UserAgentParserTest {

  UserAgentParser parser = new UserAgentParser();

  Targets modernTargets = new Targets()
    .chrome(74)
    .firefox(71)
    .edge(79)
    .safari(11, 0)
    .samsung(8, 2);

  @Test
  public void modern() {
    // unknown
    assertFalse(parser.matchTargets("Potato", modernTargets));

    //chrome
    assertTrue(parser.matchTargets("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/74.0.3729.169 Safari/537.36", modernTargets));
    assertTrue(parser.matchTargets("Mozilla/5.0 (Linux; Android 5.1.1; Nexus 5 Build/LMY48B; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/76.0.3729.169 Mobile Safari/537.36", modernTargets));
    assertFalse(parser.matchTargets("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.113 Safari/537.36", modernTargets));

    // firefox
    assertTrue(parser.matchTargets("Mozilla/5.0 (X11; Ubuntu; Linux x86_64; rv:73.0) Gecko/20100101 Firefox/73.0", modernTargets));
    assertFalse(parser.matchTargets("Mozilla/5.0 (X11; Ubuntu; Linux x86_64; rv:73.0) Gecko/20100101 Firefox/70.0", modernTargets));

    // samsung
    assertFalse(parser.matchTargets("Mozilla/5.0 (Linux; Android 8.1.0; SM-G965F Build/OPM2.171019.029) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/7.2 Chrome/59.0.3071.125 Mobile Safari/537.36", modernTargets));
    assertTrue(parser.matchTargets("Mozilla/5.0 (Linux; Android 8.1.0; SM-G965F Build/OPM2.171019.029) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/8.2 Chrome/59.0.3071.125 Mobile Safari/537.36", modernTargets));
  }

  @Test
  public void safari() {
    // safari
    Targets safari_10_1 = new Targets().safari(10, 1);
    String safariUA_10_1 = "Mozilla/5.0 (iPad; CPU OS 12_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/10.1 Mobile/15E148 Safari/604.1";
    String safariUA_10_0 = "Mozilla/5.0 (iPad; CPU OS 12_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/10.0 Mobile/15E148 Safari/604.1";
    String safariUA_11_0 = "Mozilla/5.0 (iPad; CPU OS 12_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/11.0 Mobile/15E148 Safari/604.1";
    assertFalse(parser.matchTargets(safariUA_10_0, safari_10_1));
    assertTrue(parser.matchTargets(safariUA_10_1, safari_10_1));
    assertTrue(parser.matchTargets(safariUA_11_0, safari_10_1));
  }

  @Test
  public void ie11() {
    String header = "Mozilla/5.0 CK={} (Windows NT 6.1; WOW64; Trident/7.0; rv:11.0) like Gecko";
    assertTrue(parser.matchTargets(header, new Targets().ie(11)));
  }

  /**
   * The bundled version of regexes.yaml (at time of writing) thought Edgium was Chrome, so this checks
   * that it's correctly recognised as Edge.
   */
  @Test
  public void edgium() {
    String header = "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/79.0.3945.74 Safari/537.36 Edg/79.0.309.43";
    assertTrue(parser.matchTargets(header, new Targets().edge(79)));
  }

  @Test
  public void samesiteMac() {
    String goodMacWebview = "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_5) AppleWebKit/600.8.9 (KHTML, like Gecko)";
    String badMacWebview = "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_14_5) AppleWebKit/600.8.9 (KHTML, like Gecko)";

    assertTrue(INSTANCE.isSamesiteNoneCompatible(goodMacWebview));
    assertFalse(INSTANCE.isSamesiteNoneCompatible(badMacWebview));
  }

  @Test
  public void samesiteIos() {
    String badIos = "Mozilla/5.0 (iPhone; CPU iPhone OS 12_3_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.1.1 Mobile/15E148 Safari/604.1";
    String goodIos = "Mozilla/5.0 (iPhone; CPU iPhone OS 13_4_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.1 Mobile/15E148 Safari/604.1";

    assertTrue(INSTANCE.isSamesiteNoneCompatible(goodIos));
    assertFalse(INSTANCE.isSamesiteNoneCompatible(badIos));
  }

  @Test
  public void samesiteChrome() {
    String badChrome = "Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.113 Safari/537.36";
    String goodChrome = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/74.0.3729.169 Safari/537.36";

    assertTrue(INSTANCE.isSamesiteNoneCompatible(goodChrome));
    assertFalse(INSTANCE.isSamesiteNoneCompatible(badChrome));
  }

}
