package uk.ac.warwick.util.core.spring

import org.jetbrains.annotations.Contract
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.util.FileCopyUtils
import uk.ac.warwick.util.core.StringUtils
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream
import java.text.DecimalFormat
import java.text.Normalizer
import java.util.*

/**
 * A collection of functions for dealing with the filesystem, processing filenames,
 * and formatting file sizes.
 *
 *
 */
object FileUtils {

    const val BYTE_LABEL: String = "B"
    const val KILO_BYTE_LABEL: String = "KB"
    const val MEGA_BYTE_LABEL: String = "MB"
    const val GIGA_BYTE_LABEL: String = "GB"

    private val LOGGER: Logger = LoggerFactory.getLogger(FileUtils::class.java)

    private val safeFilenameCollapsibleChars: Regex = "[\\s-]+".toRegex()

    private const val TEMP_FILE_RETRIES = 5

    private const val MAX_MB_TO_SHOW_DECIMALS = 10.0f
    private const val MAX_GB_TO_SHOW_DECIMALS = 10.0f

    private const val BYTES_IN_KB = 1024
    private const val KB_IN_MB = 1024
    private const val MB_IN_GB = 1024

    /**
     * Do a strict delete, but if the deletion fails then rename the directory to a recycle bin.
     */
    @JvmStatic
    fun recursiveDelete(file: File, deletionBin: File?) {
        recursiveDelete(file, true, deletionBin)
    }

    /**
     * Delete the tree from the specified file. The boolean strict is a
     * direction to the method on what to do when the delete fails (this
     * sometimes happens when a file handle is open, on NFS it will rename
     * rather than move the file to some safe tmp directory, so you can't delete
     * the parent directory).
     *
     * By default, deletes are strict and throw ISEs when bad things happen.
     */
    @JvmStatic
    @JvmOverloads
    fun recursiveDelete(file: File, strict: Boolean = true, deletionBin: File? = null) {
        if (file.isDirectory()) {
            if (LOGGER.isDebugEnabled()) {
                var logMessage = "Directory contains files (pre-delete):"
                logMessage += recursiveOutput(file)
                LOGGER.debug(logMessage)
            }

            for (child in file.listFiles()) {
                recursiveDelete(child, strict, deletionBin)
            }
        }
        if (LOGGER.isDebugEnabled()) LOGGER.debug("Deleting $file")
        val deletedFileOK = file.delete()
        if (!deletedFileOK) {
            if (strict) {
                // on NFS we want to move the file to some safe temporary directory
                // and mark it to be deleted on exit

                LOGGER.info("Could not delete the " + (if (file.isDirectory()) "directory" else "file") + " " + file)

                if (LOGGER.isDebugEnabled() && file.isDirectory()) {
                    if (file.list().size == 0) {
                        LOGGER.debug("Directory contains no files anymore though")
                    } else {
                        var logMessage = "Directory contains files:"
                        logMessage += recursiveOutput(file)
                        LOGGER.debug(logMessage)
                    }
                }

                if (deletionBin != null && file.isDirectory()) {
                    // rename the file to the deletion bin
                    require(!(!deletionBin.isDirectory() || !deletionBin.exists())) { "Deletion bin $deletionBin must be an existing directory" }
                    val renameToFile = File(deletionBin, file.getName() + System.currentTimeMillis())

                    check(!renameToFile.exists()) { "Could not rename directory to $renameToFile - file already exists" }
                    val success = file.renameTo(renameToFile)

                    check(success) { "Failed to rename directory to $renameToFile" }

                    renameToFile.deleteOnExit()
                } else {
                    throw IllegalStateException("Cannot delete $file")
                }
            } else {
                file.deleteOnExit()
                LOGGER.info("Could not delete the file $file, marked to delete on exit")
            }
        }
    }

    /**
     * Copies the specified source file (which must exist) into the target
     * (which must not exist). If the source is a file, then the contents of
     * that file will be copied into target. If the source is a directory, then
     * the contents of that directory will be copied into target, which will of
     * course be a directory. If recurse is true, then the directory will be
     * recursively copied.
     *
     * @param source
     * Source File or Directory.
     * @param target
     * The target.
     * @param recurse
     * Whether to recursively copy all directories.
     * @throws IOException
     */
    @JvmStatic
    @Throws(IOException::class)
    fun copy(source: File, target: File, recurse: Boolean) {
        sanityCheck(source, target, recurse)

        if (source.isDirectory()) {
            val filesInDir = source.listFiles() // *must* be taken before
            // target is created
            // in case the target is in the source directory
            if (!target.mkdirs() && !target.exists()) {
                throw IOException("Cannot create $target")
            }
            for (child in filesInDir!!) {
                if (child.isFile() || recurse) {
                    copy(child, File(target, child.getName()), recurse)
                }
            }
        } else {
            if (!target.getParentFile().exists() && !target.getParentFile().mkdirs()) {
                throw IOException("Cannot create $target")
            }
            FileCopyUtils.copy(source, target)
        }
    }

    private fun sanityCheck(source: File, target: File, recurse: Boolean) {
        require(source.exists()) { "The source $source must exist." }
        require(!(target.exists() && !target.isDirectory())) { "The target $target must not exist." }

        // if target doesn't exist .isFile() and .isDirectory() fail.
        require(!(target.exists() && source.isDirectory() != target.isDirectory())) {
            String.format(
                "Source '%s' and target '%s' must both be directories or both be files", source, target
            )
        }

        // must append a trailing slash else /calendar and /calendar-copy will
        // match
        require(
            !(recurse && target.getAbsolutePath().startsWith(source.getAbsolutePath() + "/"))
        ) { String.format("Target '%s' cannot be a descendant of source '%s'", target, source) }
    }

    /**
     * Return just the fileName. If url is "/services/its/a.b" then return
     * "a.b".
     */
    @JvmStatic
    fun getFileName(url: String): String {
        val file = File(url)
        return file.getName()
    }

    /**
     * Return everything upto the last . if there is one.
     *
     * @param s
     * @return
     */
    @JvmStatic
    fun getFileNameWithoutExtension(s: String): String {
        val indexOfLastDot = StringUtils.safeSubstring(s, 1).lastIndexOf('.')
        if (indexOfLastDot < 0) {
            return s
        }
        return s.substring(0, indexOfLastDot + 1)
    }

    private fun getExtension(s: String): String {
        val indexOfLastDot = StringUtils.safeSubstring(s, 1).lastIndexOf('.')
        if (indexOfLastDot < 0) {
            return ""
        }
        return s.substring(indexOfLastDot + 2)
    }

    /**
     * Return the extension of the specified fileName. If there is no extension,
     * this will return "".
     */
    @JvmStatic
    fun getLowerCaseExtension(filename: String): String {
        return getExtension(filename).lowercase(Locale.getDefault())
    }

    @JvmStatic
    fun extensionMatches(filename: String, extension: String): Boolean {
        val compareExtension = extension.lowercase(Locale.getDefault())
            // if the user has specified an extension like ".txt", clean it up for them
            .replace("[^.]*\\.".toRegex(), "")
        return getLowerCaseExtension(filename).equals(compareExtension, ignoreCase = true)
    }

    /**
     * Strip all invalid characters from the specified spring.
     *
     * The safety refers to the fact that only ANSI characters remain so
     * there is no need to worry about escaping, and no slashes
     * are allowed to avoid any problems with path handling.
     *
     * It removes a leading dot from the filename.
     *
     * Gaps between words are replaces with a dash, which is the recommended
     * character for SEO if these names are to be used as part of a URL path.
     *
     * Worth noting that the result is not necessarily a good filename
     * as it could be empty, have no extension, etc. so you will probably
     * need to do additional validation for your use case.
     *
     * @return A safe filename, or the original name if it was null or blank.
     */
    @JvmStatic
    @Contract("!null -> !null; null -> null")
    fun convertToSafeFileName(originalName: String?): String? {
        if (originalName.isNullOrBlank()) {
            return originalName
        }

        // Collapse accents (diacritics) to the unaccented equivalent,
        val normalised = Normalizer.normalize(originalName, Normalizer.Form.NFKD)

        // strip the path
        val file = File(normalised.lowercase(Locale.getDefault()))
        val s = file.getName()
        val fileName: String = getFileNameWithoutExtension(s)
        val extension: String = getLowerCaseExtension(s)
        val fileNameSB = StringBuffer()
        for (c in fileName.toCharArray()) {
            if (isValidForFileName(c)) {
                fileNameSB.append(c)
            }
        }

        // Shouldn't start with dots
        while (fileNameSB.isNotEmpty() && fileNameSB.get(0) == '.') {
            fileNameSB.deleteCharAt(0)
        }

        val baseName = fileNameSB.toString().trim()
        val cleanedExtension = extension.toCharArray().filter { isValidForFileName(it) }.joinToString("").trim()
        val joinedName = when {
            StringUtils.hasText(cleanedExtension) -> "$baseName.$cleanedExtension"
            else -> baseName
        }

        // replace all whitespace with "-"
        return joinedName.trim().replace(safeFilenameCollapsibleChars, "-")
    }

    private fun isValidForFileName(c: Char): Boolean {
        return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') || c == '_' || c == '-' || c == ' ' || c == '.'
    }

    /**
     * Create, or retrieve the specified file. Will create the file if it
     * doesn't already exist. Will convert any IOExceptions to runtime
     * exceptions.
     */
    @JvmStatic
    fun createOrLoadFile(parent: File, fileName: String): File {
        check(parent.isDirectory()) { "Parent $parent must be a directory" }

        val file = File(parent, fileName)
        if (!file.exists()) {
            try {
                check(file.createNewFile()) { "Cannot create file $file" }
            } catch (e: IOException) {
                throw IllegalStateException("Cannot create file $file")
            }
        }
        return file
    }

    /**
     * returns the directory structure of the file system as a string
     *
     * @param file
     * @return
     */
    @JvmStatic
    fun recursiveOutput(file: File): String {
        if (!file.exists()) {
            return ""
        }

        var output = "\n" + file.getPath()

        if (file.isDirectory()) {
            val subfiles = file.listFiles()
            for (subfile in subfiles!!) {
                output += recursiveOutput(subfile)
            }
        } else {
            output += " "
            output += file.length()
        }
        return output
    }

    @JvmStatic
    @Throws(IllegalStateException::class)
    fun createFile(theSuggestedName: String, theContents: InputStream, directory: File): File {
        val file: File = createFile(theSuggestedName, directory)

        FileOutputStream(file).use { fos ->
            try {
                FileCopyUtils.copy(theContents, fos)
            } catch (e: IOException) {
                throw IllegalStateException("cannot copy contents [$theContents] into $file", e)
            }
        }

        return file
    }

    /**
     * Create a semi-randomly named file under the given directory.
     * @param theSuggestedName The filename will not be exactly this, but it should contain it.
     * @param directory The parent directory the file will be created under
     * @return The created file.
     */
    @JvmStatic
    fun createFile(theSuggestedName: String, directory: File): File {
        check(!(!directory.exists() && !directory.mkdirs())) { "Unable to create directories for temporary file storage" }


        // If the randomly generated filename exists, keep trying with new names
        // a few times before giving up.
        repeat(TEMP_FILE_RETRIES) {
            val file = generateRandomFile(theSuggestedName, directory)
            if (!file.exists()) {
                try {
                    file.createNewFile()
                } catch (e: IOException) {
                    throw IllegalStateException("Cannot create file $file", e)
                }
                return file
            }
        }

        throw IllegalStateException("Cannot create file based on $theSuggestedName")
    }

    private fun generateRandomFile(filePrefix: String, directory: File?): File {
        var fileName: String = filePrefix + System.nanoTime()
        fileName = convertToSafeFileName(fileName)!!
        return File(directory, "$fileName.tmp")
    }

    @JvmStatic
    fun getReadableFileSize(sizeInBytes: Double): String {
        val sizeString: String?
        var units: String = BYTE_LABEL

        if (sizeInBytes < BYTES_IN_KB) {
            sizeString = "" + Math.round(sizeInBytes)
        } else {
            units = KILO_BYTE_LABEL
            val sizeInKb = sizeInBytes / BYTES_IN_KB.toDouble()

            if (sizeInKb < KB_IN_MB) {
                sizeString = "" + Math.round(sizeInKb)
            } else {
                units = MEGA_BYTE_LABEL
                val sizeInMb: Double = sizeInKb / BYTES_IN_KB
                if (sizeInMb < MB_IN_GB) {
                    sizeString = "" + roundAndFormat(sizeInMb, MAX_MB_TO_SHOW_DECIMALS)
                } else {
                    units = GIGA_BYTE_LABEL
                    val sizeInGb: Double = sizeInMb / MB_IN_GB
                    sizeString = "" + roundAndFormat(sizeInGb, MAX_GB_TO_SHOW_DECIMALS)
                }
            }
        }

        return "$sizeString $units"
    }

    private fun roundAndFormat(size: Double, maxShowDecimals: Float): String? {
        if (size < maxShowDecimals) {
            return DecimalFormat("#.0").format(size)
        }
        return "" + Math.round(size)
    }

}
