package uk.ac.warwick.util.content.texttransformers;

import java.util.Map;
import java.util.regex.Matcher;

import uk.ac.warwick.util.content.MutableContent;
import uk.ac.warwick.util.content.texttransformers.TextPatternTransformer.Callback;
import uk.ac.warwick.util.core.HtmlUtils;
import uk.ac.warwick.util.core.HttpUtils;

public final class LatexTextTransformer extends AbstractSquareTagTransformer {

    private final String latexUrl;

    private String[] ALLOWED_PARAMETERS = new String[] { "alt" };

    public LatexTextTransformer(final String theLatexUrl) {
        super("latex", true); //true = multiline
        this.latexUrl = theLatexUrl;
    }

    @Override
    protected String[] getAllowedParameters() {
        return ALLOWED_PARAMETERS;
    }

    @Override
    protected boolean isTagGeneratesHead() {
        return false;
    }

    @Override
    protected Callback getCallback() {
        return new TextPatternTransformer.Callback() {
            public String transform(final String input, final MutableContent mc) {
                Matcher matcher = getTagPattern().matcher(input);
                if (!matcher.matches()) {
                    throw new IllegalStateException(
                            "Failed to match latex tag, but shouldn't be here if it didn't");
                }

                String contents = getContents(matcher);

                if (contents == null) {
                    return input;
                }

                String urlEncodedLatex = HttpUtils.utf8Encode(contents.replaceAll(" ", "~"));
				String htmlEncodedLatex = HtmlUtils.htmlEscape(contents);

                Map<String, Object> parameters = getParameters(matcher);

                String altText = htmlEncodedLatex; // Default to this

                if (parameters.containsKey("alt")) {
                    altText = HtmlUtils.htmlEscape(parameters.get("alt").toString());
                }

				return "<notextile><img class=\"latex\" src=\"" + latexUrl
						+ "?" + urlEncodedLatex + "\" alt=\"" + altText + "\"></notextile>";
            }
        };
    }

}
