package uk.ac.warwick.util.web.useragent;

import ua_parser.Client;
import ua_parser.Parser;
import ua_parser.UserAgent;

import java.util.HashMap;
import java.util.Map;

public class UserAgentParser {

  private final Parser parser;

  public static final UserAgentParser INSTANCE = new UserAgentParser();

  public UserAgentParser() {
    // Latest regexes fetched from:
    // https://raw.githubusercontent.com/ua-parser/uap-core/master/regexes.yaml
    parser = new Parser(getClass().getResourceAsStream("/uap-core-regexes.yaml"));
  }

  public boolean matchTargets(String uaHeader, Targets targets) {
    Client result = parser.parse(uaHeader);
    for (Map.Entry<String, Version> entry : targets.getTargetVersions().entrySet()) {
      String target = entry.getKey();
      Version version = entry.getValue();
      switch (target) {
        case "chrome":
          // All kinds of Chrome (inc mobile and webview)
          if (result.userAgent.family.contains("Chrome") && versionGte(result.userAgent, version)) {
            return true;
          }
          break;

        case "firefox":
          if (result.userAgent.family.contains("Firefox") && versionGte(result.userAgent, version)) {
            return true;
          }
          break;

        case "safari":
          if (result.userAgent.family.contains("Safari") && versionGte(result.userAgent, version)) {
            return true;
          }
          break;

        case "edge":
          if (result.userAgent.family.contains("Edge") && versionGte(result.userAgent, version)) {
            return true;
          }
          break;

        case "ie":
          if (result.userAgent.family.contains("IE") && versionGte(result.userAgent, version)) {
            return true;
          }
          break;

        case "samsung":
          if (result.userAgent.family.contains("Samsung") && versionGte(result.userAgent, version)) {
            return true;
          }
          break;
      }
    }
    return false;
  }

  /**
   * Is it safe to use "None" as a SameSite value with this user agent? Some treat
   * it as Strict, others drop the cookie entirely, so we should avoid passing it to them.
   *
   * @param uaHeader User-Agent header value
   * @return whether we can pass it a SameSite=None value
   */
  public boolean isSamesiteNoneCompatible(String uaHeader) {
    Client client = parser.parse(uaHeader);
    return !failsToStrict(client) && !failsWithDrop(client);
  }

  /** Clients that DROP cookies with unrecognised samesite */
  private boolean failsWithDrop(Client client) {
    try {
      int major = Integer.parseInt(client.userAgent.major);
      switch (client.userAgent.family) {
        case "Chrome":
        case "Chromium":
          if (major >= 51 && major < 67) return true;
        case "UCBrowser":
          int minor = Integer.parseInt(client.userAgent.minor);
          int patch = Integer.parseInt(client.userAgent.patch);
          if (major < 12 && minor < 13 && patch < 2) return true;
      }
    } catch (NumberFormatException e) {
      // Some UA with a non-numerical bit of version, how strange
    }

    return false;
  }

  /** Clients that treat unrecognised samesite as "Strict" */
  private boolean failsToStrict(Client client) {
    switch (client.os.family) {
      case "iOS":
        if ("12".equals(client.os.major)) return true;
      case "Mac OS X":
        if (
            "10".equals(client.os.major) &&
            "14".equals(client.os.minor) &&
            (
              "Safari".equals(client.userAgent.family) ||
              "Apple Mail".equals(client.userAgent.family) // any embedded mac web view apparently is Apple Mail
            )
        ) return true;
    }

    return false;
  }

  private boolean versionGte(UserAgent ua, Version version) {
    try {
      int uaMajor = Integer.parseInt(ua.major);
      boolean gtMajor = uaMajor > version.major;
      boolean eqMajor = uaMajor >= version.major;
      return gtMajor || (eqMajor && Integer.parseInt(ua.minor) >= version.minor);
    } catch (NumberFormatException e) {
      return false;
    }
  }

}

class Version {
  public final int major;
  public final int minor;
  public Version(int major, int minor) {
    this.major = major;
    this.minor = minor;
  }
}
