package uk.ac.warwick.util.termdates;

import com.google.common.collect.ImmutableList;

import java.time.Month;
import java.time.YearMonth;
import java.time.temporal.Temporal;
import java.util.List;
import java.util.regex.Matcher;

import static java.lang.Integer.*;
import static java.util.Collections.*;

public class ExtendedAcademicYear extends AcademicYear {

    /**
     * The strategy for how far to extend the summer vacation of the academic year beyond 31st July
     */
    public enum ExtendedEndStrategy {
        /**
         * End the extended academic year before Monday of week 1 of the following academic year
         */
        StartOfAutumnTerm,

        /**
         * End the extended academic year at the end of the autumn term of the following academic year
         */
        EndOfAutumnTerm
    }

    private final ExtendedEndStrategy strategy;

    private ExtendedAcademicYear(int startYear, List<AcademicYearPeriod> periods, ExtendedEndStrategy strategy) {
        super(startYear, periods);
        this.strategy = strategy;
    }

    static ExtendedAcademicYear wrap(AcademicYear year, AcademicYear next, ExtendedEndStrategy strategy) {
        if (year.isPlaceholder() || next.isPlaceholder()) {
            return placeholder(year.getStartYear(), strategy);
        }

        // Extend the summer vacation
        ImmutableList.Builder<AcademicYearPeriod> periods = ImmutableList.builder();
        for (AcademicYearPeriod period : year.getPeriods()) {
            if (period.getType() == AcademicYearPeriod.PeriodType.summerVacation) {
                switch (strategy) {
                    case StartOfAutumnTerm:
                        periods.add(
                            Vacation.between(
                                AcademicYearPeriod.PeriodType.summerVacation,
                                (Term) year.getPeriod(AcademicYearPeriod.PeriodType.summerTerm),
                                (Term) next.getPeriod(AcademicYearPeriod.PeriodType.autumnTerm)
                            )
                        );
                        break;
                    case EndOfAutumnTerm:
                        periods.add(
                            Vacation.between(
                                AcademicYearPeriod.PeriodType.summerVacation,
                                (Term) year.getPeriod(AcademicYearPeriod.PeriodType.summerTerm),
                                next.getPeriod(AcademicYearPeriod.PeriodType.autumnTerm).getLastDay()
                            )
                        );
                        break;
                    default:
                        throw new IllegalArgumentException("Unexpected ExtendedEndStrategy " + strategy);
                }
            } else {
                periods.add(period);
            }
        }

        return new ExtendedAcademicYear(year.getStartYear(), periods.build(), strategy);
    }

    static ExtendedAcademicYear placeholder(int startYear, ExtendedEndStrategy strategy) {
        return new ExtendedAcademicYear(startYear, emptyList(), strategy);
    }

    /**
     * @deprecated Use {@link #starting(int, ExtendedEndStrategy)} specifying the ExtendedEndStrategy explicitly
     */
    public static ExtendedAcademicYear starting(int startYear) {
        return starting(startYear, ExtendedEndStrategy.StartOfAutumnTerm);
    }

    public static ExtendedAcademicYear starting(int startYear, ExtendedEndStrategy strategy) {
        return TermDatesService.INSTANCE.getExtendedAcademicYear(startYear, strategy);
    }

    /**
     * @deprecated Use {@link #forDate(Temporal, ExtendedEndStrategy)} specifying the ExtendedEndStrategy explicitly
     */
    public static ExtendedAcademicYear forDate(Temporal temporal) {
        return forDate(temporal, ExtendedEndStrategy.StartOfAutumnTerm);
    }

    public static ExtendedAcademicYear forDate(Temporal temporal, ExtendedEndStrategy strategy) {
        YearMonth yearMonth = YearMonth.from(temporal);
        return (yearMonth.getMonthValue() < Month.AUGUST.getValue()) ? starting(yearMonth.getYear() - 1, strategy) : starting(yearMonth.getYear(), strategy);
    }

    /**
     * @deprecated Use {@link #parse(String, ExtendedEndStrategy)} specifying the ExtendedEndStrategy explicitly
     */
    public static ExtendedAcademicYear parse(String pattern) {
        return parse(pattern, ExtendedEndStrategy.StartOfAutumnTerm);
    }

    public static ExtendedAcademicYear parse(String pattern, ExtendedEndStrategy strategy) {
        Matcher m = SITS_PATTERN.matcher(pattern);

        if (m.matches()) {
            int startYear = parseInt(m.group(1));
            return (startYear > CENTURY_BREAK) ? starting(1900 + startYear, strategy) : starting(2000 + startYear, strategy);
        } else {
            throw new IllegalArgumentException("Did not match YY/YY: " + pattern);
        }
    }

    @Override
    public ExtendedAcademicYear previous() {
        return starting(getStartYear() - 1, strategy);
    }

    @Override
    public ExtendedAcademicYear next() {
        return starting(getStartYear() + 1, strategy);
    }

    @Override
    public List<AcademicYear> yearsSurrounding(int yearsBefore, int yearsAfter) {
        verify(yearsBefore >= 0 && yearsAfter >= 0);
        ImmutableList.Builder<AcademicYear> years = ImmutableList.builder();

        for (int year = getStartYear() - yearsBefore; year <= getStartYear() + yearsAfter; year++) {
            // Broken out into a variable so we can guarantee it's an ExtendedAcademicYear
            ExtendedAcademicYear yearToAdd = starting(year, strategy);
            years.add(yearToAdd);
        }

        return years.build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ExtendedAcademicYear that = (ExtendedAcademicYear) o;
        return super.equals(that) && this.strategy == that.strategy;
    }
}
