package uk.ac.warwick.util.termdates;

import org.apache.commons.lang3.NotImplementedException;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.threeten.extra.LocalDateRange;

import java.io.Serializable;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;

public class AcademicWeek implements Comparable<AcademicWeek>, Serializable {

    private final AcademicYear year;

    private final AcademicYearPeriod period;

    private final int weekNumber;

    private final LocalDateRange dateRange;

    private AcademicWeek(AcademicYear year, AcademicYearPeriod period, int weekNumber, LocalDateRange dateRange) {
        this.year = year;
        this.period = period;
        this.weekNumber = weekNumber;
        this.dateRange = dateRange;
    }

    static AcademicWeek of(AcademicYear academicYear, AcademicYearPeriod period, int weekNumber, LocalDateRange dateRange) {
        return new AcademicWeek(academicYear, period, weekNumber, dateRange);
    }

    public AcademicYear getYear() {
        return year;
    }

    public AcademicYearPeriod getPeriod() {
        return period;
    }

    /**
     * Week 1 is from the first day of the autumn term, any week before that may be 0 or negative.
     */
    public int getWeekNumber() {
        return weekNumber;
    }

    /**
     * The week within a term, starting at 1 for the week of the first day of term.
     *
     * @throws IllegalStateException if the week is outside of term time
     */
    public int getTermWeekNumber() {
        if (!period.isTerm()) throw new IllegalStateException();

        int number = 1;
        LocalDate startDate = period.getFirstDay();
        while (!dateRange.contains(startDate)) {
            number++;
            startDate = startDate.plusWeeks(1);
        }

        return number;
    }

    /**
     * The week within a term, cumulatively counting all term weeks for the year. For the first week of spring term,
     * this would be 11 (10 weeks of autumn term + 1 week of spring term)
     *
     * @throws IllegalStateException if the week is outside of term time
     */
    public int getCumulativeWeekNumber() {
        if (!period.isTerm()) throw new IllegalStateException();

        final int modifier;
        switch (period.getType()) {
            case springTerm:
                modifier = 10;
                break;
            case summerTerm:
                modifier = 20;
                break;
            default:
                modifier = 0;
        }

        return modifier + getTermWeekNumber();
    }

    /**
     * An alternative numbering system to academic week numbering; week 1 is the first Monday on or after 1st August.
     *
     * SITS doesn't include an AYW record for the (partial) week before the first Monday of August, but this will return
     * 0 for that week.
     */
    public int getSitsWeekNumber() {
        // Get the week offset between SITS and academic week numbers by looking how many weeks back the first Monday of
        // August is
        LocalDate firstMondayInAugust =
            year.getPeriods().iterator().next()                          // First period of year (pre-term vacation)
                .getFirstDay()                                           // First day (i.e. August 1st)
                .with(TemporalAdjusters.firstInMonth(DayOfWeek.MONDAY)); // First Monday of month

        LocalDate firstDayOfAutumnTerm =
            year.getPeriod(AcademicYearPeriod.PeriodType.autumnTerm)
                .getFirstDay()
                .with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY));

        int offset = (int) ChronoUnit.WEEKS.between(firstMondayInAugust, firstDayOfAutumnTerm);
        return weekNumber + offset;
    }

    public LocalDateRange getDateRange() {
        return dateRange;
    }

    @Override
    public int compareTo(AcademicWeek o) {
        int result = this.year.compareTo(o.year);
        if (result != 0) return result;

        result = this.period.compareTo(o.period);
        if (result != 0) return result;

        return this.weekNumber - o.weekNumber;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;

        if (o == null || getClass() != o.getClass()) return false;

        AcademicWeek that = (AcademicWeek) o;

        return new EqualsBuilder()
            .append(weekNumber, that.weekNumber)
            .append(year, that.year)
            .append(period, that.period)
            .isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37)
            .append(year)
            .append(period)
            .append(weekNumber)
            .toHashCode();
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE)
            .append("year", year)
            .append("period", period)
            .append("weekNumber", weekNumber)
            .append("dateRange", dateRange)
            .toString();
    }
}
