package uk.ac.warwick.util.files.impl;

import com.google.common.annotations.Beta;
import org.jclouds.blobstore.BlobStore;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.domain.*;
import org.jclouds.blobstore.options.*;
import org.jclouds.domain.Location;
import org.jclouds.io.Payload;
import org.jclouds.javax.annotation.Nullable;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;

/**
 * Wrapper for a {@link BlobStore} that translates the container name to one
 * that is acceptable to the underlying blob store.
 * 
 * Translation may not be perfect, e.g. blob.getContainer() will return the
 * real untranslated container name.
 */
public class ContainerRenamingBlobStore implements BlobStore {
  private final BlobStoreContext context;
  private final RenameMode mode;

  @Override
  public BlobStoreContext getContext() {
    return delegate.getContext();
  }

  @Override
  public BlobBuilder blobBuilder(String name) {
    return delegate.blobBuilder(name);
  }

  @Override
  public Set<? extends Location> listAssignableLocations() {
    return delegate.listAssignableLocations();
  }

  @Override
  public PageSet<? extends StorageMetadata> list() {
    return delegate.list();
  }

  @Override
  public boolean containerExists(String container) {
    return delegate.containerExists(rename(container));
  }

  @Override
  public boolean createContainerInLocation(Location location, String container) {
    return delegate.createContainerInLocation(location, rename(container));
  }

  @Override
  public boolean createContainerInLocation(Location location, String container, CreateContainerOptions options) {
    return delegate.createContainerInLocation(location, rename(container), options);
  }

  @Override
  @Beta
  public ContainerAccess getContainerAccess(String container) {
    return delegate.getContainerAccess(rename(container));
  }

  @Override
  @Beta
  public void setContainerAccess(String container, ContainerAccess access) {
    delegate.setContainerAccess(rename(container), access);
  }

  @Override
  public PageSet<? extends StorageMetadata> list(String container) {
    return delegate.list(rename(container));
  }

  @Override
  public PageSet<? extends StorageMetadata> list(String container, ListContainerOptions options) {
    return delegate.list(rename(container), options);
  }

  @Override
  public void clearContainer(String container) {
    delegate.clearContainer(rename(container));
  }

  @Override
  public void clearContainer(String container, ListContainerOptions options) {
    delegate.clearContainer(rename(container), options);
  }

  @Override
  public void deleteContainer(String container) {
    delegate.deleteContainer(rename(container));
  }

  @Override
  public boolean deleteContainerIfEmpty(String container) {
    return delegate.deleteContainerIfEmpty(rename(container));
  }

  @Override
  @Deprecated
  public boolean directoryExists(String container, String directory) {
    return delegate.directoryExists(rename(container), directory);
  }

  @Override
  @Deprecated
  public void createDirectory(String container, String directory) {
    delegate.createDirectory(rename(container), directory);
  }

  @Override
  @Deprecated
  public void deleteDirectory(String containerName, String name) {
    delegate.deleteDirectory(containerName, name);
  }

  @Override
  public boolean blobExists(String container, String name) {
    return delegate.blobExists(rename(container), name);
  }

  @Override
  public String putBlob(String container, Blob blob) {
    return delegate.putBlob(rename(container), blob);
  }

  @Override
  public String putBlob(String container, Blob blob, PutOptions options) {
    return delegate.putBlob(rename(container), blob, options);
  }

  @Override
  @Beta
  public String copyBlob(String fromContainer, String fromName, String toContainer, String toName, CopyOptions options) {
    return delegate.copyBlob(rename(fromContainer), fromName, rename(toContainer), toName, options);
  }

  @Override
  @Nullable
  public BlobMetadata blobMetadata(String container, String name) {
    return delegate.blobMetadata(rename(container), name);
  }

  @Override
  @Nullable
  public Blob getBlob(String container, String name) {
    return delegate.getBlob(rename(container), name);
  }

  @Override
  @Nullable
  public Blob getBlob(String container, String name, GetOptions options) {
    return delegate.getBlob(rename(container), name, options);
  }

  @Override
  public void removeBlob(String container, String name) {
    delegate.removeBlob(rename(container), name);
  }

  @Override
  public void removeBlobs(String container, Iterable<String> names) {
    delegate.removeBlobs(rename(container), names);
  }

  @Override
  @Beta
  public BlobAccess getBlobAccess(String container, String name) {
    return delegate.getBlobAccess(rename(container), name);
  }

  @Override
  @Beta
  public void setBlobAccess(String container, String name, BlobAccess access) {
    delegate.setBlobAccess(rename(container), name, access);
  }

  @Override
  public long countBlobs(String container) {
    return delegate.countBlobs(rename(container));
  }

  @Override
  public long countBlobs(String container, ListContainerOptions options) {
    return delegate.countBlobs(rename(container), options);
  }

  @Override
  @Beta
  public MultipartUpload initiateMultipartUpload(String container, BlobMetadata blob, PutOptions options) {
    return delegate.initiateMultipartUpload(rename(container), blob, options);
  }

  @Override
  @Beta
  public void abortMultipartUpload(MultipartUpload mpu) {
    delegate.abortMultipartUpload(mpu);
  }

  @Override
  @Beta
  public String completeMultipartUpload(MultipartUpload mpu, List<MultipartPart> parts) {
    return delegate.completeMultipartUpload(mpu, parts);
  }

  @Override
  @Beta
  public MultipartPart uploadMultipartPart(MultipartUpload mpu, int partNumber, Payload payload) {
    return delegate.uploadMultipartPart(mpu, partNumber, payload);
  }

  @Override
  @Beta
  public List<MultipartPart> listMultipartUpload(MultipartUpload mpu) {
    return delegate.listMultipartUpload(mpu);
  }

  @Override
  @Beta
  public List<MultipartUpload> listMultipartUploads(String container) {
    return delegate.listMultipartUploads(rename(container));
  }

  @Override
  @Beta
  public long getMinimumMultipartPartSize() {
    return delegate.getMinimumMultipartPartSize();
  }

  @Override
  @Beta
  public long getMaximumMultipartPartSize() {
    return delegate.getMaximumMultipartPartSize();
  }

  @Override
  @Beta
  public int getMaximumNumberOfParts() {
    return delegate.getMaximumNumberOfParts();
  }

  @Override
  @Beta
  public void downloadBlob(String container, String name, File destination) {
    delegate.downloadBlob(rename(container), name, destination);
  }

  @Override
  @Beta
  public void downloadBlob(String container, String name, File destination, ExecutorService executor) {
    delegate.downloadBlob(rename(container), name, destination, executor);
  }

  @Override
  @Beta
  public InputStream streamBlob(String container, String name) {
    return delegate.streamBlob(rename(container), name);
  }

  @Override
  @Beta
  public InputStream streamBlob(String container, String name, ExecutorService executor) {
    return delegate.streamBlob(rename(container), name, executor);
  }

  private final BlobStore delegate;

  public ContainerRenamingBlobStore(BlobStoreContext context, BlobStore blobStore, RenameMode mode) {
    this.delegate = blobStore;
    this.context = context;
    this.mode = mode;
  }
  
  private String rename(String containerName) {
    return mode.rename(containerName);
  }
  
}

