package uk.ac.warwick.util.files.imageresize;

import com.google.common.io.ByteSource;
import org.apache.commons.imaging.formats.tiff.constants.TiffTagConstants;
import uk.ac.warwick.util.files.hash.HashString;
import uk.ac.warwick.util.files.imageresize.ImageIoResizer.NotAnImageException;

import java.io.IOException;
import java.io.OutputStream;
import java.time.ZonedDateTime;

public interface ImageResizer {
    enum FileType { png, gif, jpg, webp }

    enum Orientation {
        Normal(false, 0),              // EXIF 1
        Mirrored(true, 0),             // EXIF 2
        Rotate180(false, 180),         // EXIF 3
        MirroredVertically(true, 180), // EXIF 4
        MirroredRotate270(true, 270),  // EXIF 5
        Rotate90(false, 90),           // EXIF 6
        MirroredRotate90(true, 90),    // EXIF 7
        Rotate270(false, 270);         // EXIF 8

        public final boolean mirrored;
        public final int rotationDegrees;

        Orientation(boolean mirrored, int rotationDegrees) {
            this.mirrored = mirrored;
            this.rotationDegrees = rotationDegrees;
        }

        static Orientation fromExifOrientationValue(int value) {
            switch (value) {
                case TiffTagConstants.ORIENTATION_VALUE_MIRROR_HORIZONTAL:
                    return Mirrored;
                case TiffTagConstants.ORIENTATION_VALUE_ROTATE_180:
                    return Rotate180;
                case TiffTagConstants.ORIENTATION_VALUE_MIRROR_VERTICAL:
                    return MirroredVertically;
                case TiffTagConstants.ORIENTATION_VALUE_MIRROR_HORIZONTAL_AND_ROTATE_270_CW:
                    return MirroredRotate270;
                case TiffTagConstants.ORIENTATION_VALUE_ROTATE_90_CW:
                    return Rotate90;
                case TiffTagConstants.ORIENTATION_VALUE_MIRROR_HORIZONTAL_AND_ROTATE_90_CW:
                    return MirroredRotate90;
                case TiffTagConstants.ORIENTATION_VALUE_ROTATE_270_CW:
                    return Rotate270;
                case TiffTagConstants.ORIENTATION_VALUE_HORIZONTAL_NORMAL:
                default:
                    return Normal;
            }
        }
    }

    /**
     * Attempt to write a resized version of an image based on maxWidth and maxHeight.
     *
     * The input image will be read using the reader for the requested fileType.
     *
     * The output image will be in the same format as the input.
     *
     * @param source The source image.
     * @param hash Unique content identifier for the image. Not used for resizing but may be used by caching wrappers.
     * @param entityLastModified Used for caching.
     * @param out The OutputStream that the result is written to.
     * @param maxWidth If >0, a maximum width for the output image. If 0, no maximum width is enforced.
     * @param maxHeight If >0, a maximum height for the output image. If 0, no maximum height is enforced.
     * @param fileType The file type of the source image. It is currently up to the caller to accurately know what the actual
     *                 image format is, and the resizer will use that specific format reader even if the source data is actually
     *                 a different format.
     * @throws IOException If there is an I/O related problem
     * @throws NotAnImageException If the input can't be read as any known image format.
     */
    void renderResized(final ByteSource source, final HashString hash, final ZonedDateTime entityLastModified, final OutputStream out, final int maxWidth, final int maxHeight, final FileType fileType)
            throws IOException;

    long getResizedImageLength(final ByteSource source, final HashString hash, final ZonedDateTime entityLastModified, final int maxWidth, final int maxHeight, final FileType fileType) throws IOException;

}
