package uk.ac.warwick.util.mywarwick;

import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.jmock.Expectations;
import org.jmock.Mockery;
import org.jmock.integration.junit4.JUnit4Mockery;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.quartz.*;
import uk.ac.warwick.util.mywarwick.model.Instance;
import uk.ac.warwick.util.mywarwick.model.TestConfiguration;
import uk.ac.warwick.util.mywarwick.model.response.Error;
import uk.ac.warwick.util.mywarwick.model.response.Response;

import java.net.SocketTimeoutException;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

public class SendMyWarwickActivityJobTest {

    private final Mockery m = new JUnit4Mockery();

    private final MyWarwickService myWarwickService = m.mock(MyWarwickService.class);
    private final Scheduler scheduler = m.mock(Scheduler.class);
    private JobExecutionContext context;
    private JobDataMap jobDataMap;

    private final Instance prodInstance = new Instance("https://my.warwick.ac.uk", null, null, null, null);
    private final Instance testInstance = new Instance("https://my-test.warwick.ac.uk", null, null, null, null);
    private SendMyWarwickActivityJob job;

    private final Trigger trigger = m.mock(Trigger.class);
    private final TriggerKey triggerKey = new TriggerKey(UUID.randomUUID().toString(), SendMyWarwickActivityJob.JOB_KEY.getGroup());
    private CompletableFuture<Response> future;

    @Before
    public void setUp() {
        Set<Instance> instances = new HashSet<>();
        instances.add(prodInstance);
        instances.add(testInstance);

        job = new SendMyWarwickActivityJob(myWarwickService, scheduler, new TestConfiguration() {
            @Override
            public Set<Instance> getInstances() {
                return instances;
            }
        });

        context = m.mock(JobExecutionContext.class);
        jobDataMap = new JobDataMap();
        jobDataMap.put(SendMyWarwickActivityJob.INSTANCE_BASE_URL_DATA_KEY, "https://my-test.warwick.ac.uk");
        jobDataMap.put(SendMyWarwickActivityJob.REQUEST_BODY_JOB_DATA_KEY, "{}");
        jobDataMap.putAsString(SendMyWarwickActivityJob.IS_NOTIFICATION_JOB_DATA_KEY, true);
        jobDataMap.putAsString(SendMyWarwickActivityJob.IS_TRANSIENT_JOB_DATA_KEY, false);

        future = new CompletableFuture<>();
    }

    /**
     * https://bugs.elab.warwick.ac.uk/browse/UTL-258
     */
    @Test
    public void reschedulesOnFailedFuture() throws Exception {
        future.completeExceptionally(new SocketTimeoutException());

        expectReschedule();

        job.execute(context);

        // Make sure reschedule has been called
        m.assertIsSatisfied();
    }

    @Test
    public void reschedulesOnCancelledFuture() throws Exception {
        future.cancel(true);

        expectReschedule();

        job.execute(context);

        // Make sure reschedule has been called
        m.assertIsSatisfied();
    }

    @Test
    public void retriesOnPermissionValidationError() throws Exception {
        Response value = new Response();
        value.setSuccess(false);
        Error error = new Error();
        error.setId("no-permission");
        error.setMessage("You forgot to grant the permission, silly!");
        value.setError(error);
        future.complete(value);

        expectReschedule();

        job.execute(context);

        // Make sure reschedule has been called
        m.assertIsSatisfied();
    }

    @Test
    public void cancelsOnValidationFailure() throws Exception {
        Response value = new Response();
        value.setSuccess(false);
        Error error = new Error();
        error.setId("bad-thing");
        error.setMessage("The request was invalid and retrying won't help");
        value.setError(error);
        future.complete(value);

        expectNoReschedule();

        job.execute(context);

        m.assertIsSatisfied();
    }

    @Test
    public void reschedulesOnNullSuccess() throws Exception {
        future.complete(new Response()); // handle as many null values as possible

        expectReschedule();

        job.execute(context);

        // Make sure reschedule has been called
        m.assertIsSatisfied();
    }

    @Test
    public void notRescheduleOnSuccess() throws Exception {
        Response value = new Response();
        value.setSuccess(true);
        future.complete(value);

        expectNoReschedule();

        job.execute(context);

        m.assertIsSatisfied();
    }

    @Test
    public void transitionsJobDataForInstancelessInvocations() throws Exception {
        jobDataMap.clear();
        jobDataMap.put(SendMyWarwickActivityJob.REQUEST_BODY_JOB_DATA_KEY, "{}");
        jobDataMap.putAsString(SendMyWarwickActivityJob.IS_NOTIFICATION_JOB_DATA_KEY, true);
        jobDataMap.putAsString(SendMyWarwickActivityJob.IS_TRANSIENT_JOB_DATA_KEY, false);

        m.checking(new Expectations() {{
            atLeast(1).of(context).getMergedJobDataMap(); will(returnValue(jobDataMap));

            // For scheduling new jobs
            atLeast(1).of(context).getTrigger(); will(returnValue(trigger));
            atLeast(1).of(trigger).getJobDataMap(); will(returnValue(jobDataMap));

            oneOf(scheduler).scheduleJob(with(new BaseMatcher<Trigger>() {
                @Override
                public boolean matches(Object o) {
                    if (!(o instanceof Trigger)) return false;
                    Trigger t = (Trigger) o;
                    return t.getJobDataMap().getString(SendMyWarwickActivityJob.INSTANCE_BASE_URL_DATA_KEY).equals(prodInstance.getBaseUrl());
                }

                @Override
                public void describeTo(Description description) {
                    description.appendText("a trigger for prodInstance");
                }
            }));
            oneOf(scheduler).scheduleJob(with(new BaseMatcher<Trigger>() {
                @Override
                public boolean matches(Object o) {
                    if (!(o instanceof Trigger)) return false;
                    Trigger t = (Trigger) o;
                    return t.getJobDataMap().getString(SendMyWarwickActivityJob.INSTANCE_BASE_URL_DATA_KEY).equals(testInstance.getBaseUrl());
                }

                @Override
                public void describeTo(Description description) {
                    description.appendText("a trigger for testInstance");
                }
            }));

            never(myWarwickService);
        }});

        job.execute(context);

        m.assertIsSatisfied();
    }

    // Just a catch all in case we forget to run this in a test
    @After
    public void assertIsSatisfied() {
        m.assertIsSatisfied();
    }

    public void expectReschedule() throws Exception {
        maybeExpectReschedule(true);
    }

    public void expectNoReschedule() throws Exception {
        maybeExpectReschedule(false);
    }

    private void maybeExpectReschedule(boolean reschedule) throws Exception{
        int invocationCount = reschedule ? 1 : 0;

        m.checking(new Expectations() {{
            atLeast(1).of(context).getMergedJobDataMap(); will(returnValue(jobDataMap));
            oneOf(myWarwickService).sendSingleInstance(testInstance, "{}", true, false, 3); will(returnValue(future));

            // For re-scheduling
            atLeast(invocationCount).of(context).getTrigger(); will(returnValue(trigger));
            atLeast(invocationCount).of(trigger).getKey(); will(returnValue(triggerKey));
            atLeast(invocationCount).of(trigger).getJobDataMap(); will(returnValue(jobDataMap));

            exactly(invocationCount).of(scheduler).rescheduleJob(with(equal(triggerKey)), with(aNonNull(Trigger.class)));
        }});
    }
}
