package uk.ac.warwick.util.mywarwick.healthcheck;

import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import uk.ac.warwick.util.core.DateTimeUtils;
import uk.ac.warwick.util.mywarwick.SendMyWarwickActivityJob;
import uk.ac.warwick.util.mywarwick.model.Configuration;
import uk.ac.warwick.util.service.ServiceHealthcheck;
import uk.ac.warwick.util.service.ServiceHealthcheckProvider;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;

@Singleton
public abstract class AbstractMyWarwickQuartzHealthcheckProvider extends ServiceHealthcheckProvider {

    static final long INITIAL_DELAY_MS = 0L;

    static final long RUN_INTERVAL_MS = 60 * 1000L; // 1 Minute

    private final Scheduler scheduler;

    private final String name;

    private final long queueSizeWarningThreshold;

    private final long queueSizeCriticalThreshold;

    private final Duration delayWarningThreshold;

    private final Duration delayCriticalThreshold;

    @Inject
    public AbstractMyWarwickQuartzHealthcheckProvider(Scheduler scheduler, Configuration configuration) {
        super(new ServiceHealthcheck(configuration.getQuartzHealthcheckName(), ServiceHealthcheck.Status.Unknown, LocalDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION)));

        this.scheduler = scheduler;
        this.name = configuration.getQuartzHealthcheckName();
        this.queueSizeWarningThreshold = configuration.getQuartzHealthcheckQueueSizeWarningThreshold();
        this.queueSizeCriticalThreshold = configuration.getQuartzHealthcheckQueueSizeCriticalThreshold();
        this.delayWarningThreshold = configuration.getQuartzHealthcheckDelayWarningThreshold();
        this.delayCriticalThreshold = configuration.getQuartzHealthcheckDelayCriticalThreshold();
    }

    @Override
    public void run() {
        try {
            List<? extends Trigger> triggers = scheduler.getTriggersOfJob(SendMyWarwickActivityJob.JOB_KEY);

            long queueSize = 0;
            OffsetDateTime oldestNotification = OffsetDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION);
            for (Trigger trigger: triggers) {
                Trigger.TriggerState triggerState = scheduler.getTriggerState(trigger.getKey());

                if (triggerState == Trigger.TriggerState.NORMAL) {
                    queueSize += 1;
                }

                if (triggerState != Trigger.TriggerState.COMPLETE) {
                    String createdDateTimeISO8601 = trigger.getJobDataMap().getString(SendMyWarwickActivityJob.CREATED_DATETIME_ISO8601_DATA_KEY);
                    if (createdDateTimeISO8601 != null) {
                        OffsetDateTime createdDateTime = OffsetDateTime.parse(createdDateTimeISO8601);
                        if (createdDateTime.isBefore(oldestNotification)) {
                            oldestNotification = createdDateTime;
                        }
                    }
                }
            }

            OffsetDateTime delayCriticalThresholdDT = OffsetDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION).minus(delayCriticalThreshold);
            OffsetDateTime delayWarningThresholdDT = OffsetDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION).minus(delayWarningThreshold);

            final ServiceHealthcheck.Status status;
            if (queueSize >= queueSizeCriticalThreshold || oldestNotification.isBefore(delayCriticalThresholdDT)) {
                status = ServiceHealthcheck.Status.Error;
            } else if (queueSize >= queueSizeWarningThreshold || oldestNotification.isBefore(delayWarningThresholdDT)) {
                status = ServiceHealthcheck.Status.Warning;
            } else {
                status = ServiceHealthcheck.Status.Okay;
            }

            final String message;
            if (queueSize == 0) {
                message = "No queued activities";
            } else {
                message = String.format(
                    "%d queued My Warwick activities%s; oldest activity %d minutes old%s",
                    queueSize,
                    (queueSize >= queueSizeCriticalThreshold ? " (!!)" : (queueSize >= queueSizeWarningThreshold ? " (!)" : "")),
                    ChronoUnit.MINUTES.between(oldestNotification, OffsetDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION)),
                    (oldestNotification.isBefore(delayCriticalThresholdDT) ? " (!!)" : (oldestNotification.isBefore(delayWarningThresholdDT) ? " (!)" : ""))
                );
            }

            update(new ServiceHealthcheck(
                name,
                status,
                LocalDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION),
                message,
                Arrays.asList(
                    new ServiceHealthcheck.PerformanceData<>("queue_size", queueSize, queueSizeWarningThreshold, queueSizeCriticalThreshold),
                    new ServiceHealthcheck.PerformanceData<>(
                        "delay_minutes",
                        ChronoUnit.MINUTES.between(oldestNotification, OffsetDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION)),
                        delayWarningThreshold.toMinutes(),
                        delayCriticalThreshold.toMinutes()
                    )
                )
            ));
        } catch (SchedulerException e) {
            update(new ServiceHealthcheck(
                name,
                ServiceHealthcheck.Status.Unknown,
                LocalDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION),
                "Unable to get My Warwick activity queue information; " + e.getMessage()
            ));
        }
    }
}
