package uk.ac.warwick.util.mywarwick;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.quartz.JobBuilder;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.TriggerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.warwick.util.core.DateTimeUtils;
import uk.ac.warwick.util.mywarwick.model.Configuration;
import uk.ac.warwick.util.mywarwick.model.Instance;
import uk.ac.warwick.util.mywarwick.model.request.Activity;
import uk.ac.warwick.util.mywarwick.model.request.PushNotification;
import uk.ac.warwick.util.mywarwick.model.response.Response;

import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import javax.validation.constraints.NotNull;
import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.util.Base64;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

@Named
@Singleton
public class MyWarwickServiceImpl implements MyWarwickService {

    private final Logger LOGGER = LoggerFactory.getLogger(MyWarwickServiceImpl.class);
    private final Set<Instance> instances;
    private final HttpClient httpclient;
    private final ObjectMapper mapper = new ObjectMapper();
    private final MyWarwickHttpResponseCallbackHelper callbackHelper = new DefaultMyWarwickHttpResponseCallbackHelper();

    @Inject
    public MyWarwickServiceImpl(HttpClient httpclient, Configuration configuration) {
        configuration.validate();

        this.httpclient = httpclient;
        this.instances = configuration.getInstances();

        httpclient.start();
    }

    private CompletableFuture<List<Response>> sendImmediately(Activity activity, boolean isNotification, boolean isTransient, int maxAttempts) {
        return sendImmediately(makeJsonBody(activity), isNotification, isTransient, maxAttempts);
    }

    private CompletableFuture<List<Response>> sendImmediately(String requestJson, boolean isNotification, boolean isTransient, int maxAttempts) {
        List<CompletableFuture<Response>> listOfCompletableFutures = instances.stream().map(instance ->
            sendSingleInstance(instance, requestJson, isNotification, isTransient, maxAttempts)
        ).collect(Collectors.toList());

        return CompletableFuture.allOf(listOfCompletableFutures.toArray(new CompletableFuture[0]))
            .thenApply(v -> listOfCompletableFutures
                .stream()
                .map(CompletableFuture::join)
                .collect(Collectors.toList())
            );
    }

    @Override
    public CompletableFuture<Response> sendSingleInstance(Instance instance, String requestJson, boolean isNotification, boolean isTransient, int maxAttempts) {
        CompletableFuture<Response> completableFuture = new CompletableFuture<>();
        final String reqPath;
        if (isTransient && isNotification) {
            reqPath = instance.getTransientPushPath();
        } else if (isNotification) {
            reqPath = instance.getNotificationPath();
        } else {
            reqPath = instance.getActivityPath();
        }

        HttpPost request = makeRequest(
            reqPath,
            requestJson,
            instance.getApiUser(),
            instance.getApiPassword(),
            instance.getProviderId()
        );

        MyWarwickHttpResponseCallback myWarwickHttpResponseCallback = new MyWarwickHttpResponseCallback(
            reqPath,
            requestJson,
            instance,
            LOGGER,
            completableFuture,
            callbackHelper
        );

        RetryingHttpResponseCallback retryingHttpResponseCallback = new RetryingHttpResponseCallback(
            httpclient,
            request,
            myWarwickHttpResponseCallback,
            LOGGER,
            maxAttempts
        );

        httpclient.execute(request, retryingHttpResponseCallback);
        return completableFuture;
    }

    @Override
    public CompletableFuture<List<Response>> sendAsActivity(Activity activity) {
        return sendImmediately(activity, false, false, DEFAULT_MAX_ATTEMPTS);
    }

    @Override
    public CompletableFuture<List<Response>> sendAsActivity(Activity activity, int maxAttempts) {
        return sendImmediately(activity, false, false, maxAttempts);
    }

    @Override
    public CompletableFuture<List<Response>> sendAsNotification(Activity activity) {
        return sendImmediately(activity, true, false, DEFAULT_MAX_ATTEMPTS);
    }

    @Override
    public CompletableFuture<List<Response>> sendAsNotification(Activity activity, int maxAttempts) {
        return sendImmediately(activity, true, false, maxAttempts);
    }

    @Override
    public CompletableFuture<List<Response>> sendAsTransientPush(PushNotification pushNotification) {
        return sendImmediately(pushNotification, true, true, DEFAULT_MAX_ATTEMPTS);
    }

    @Override
    public CompletableFuture<List<Response>> sendAsTransientPush(PushNotification pushNotification, int maxAttempts) {
        return sendImmediately(pushNotification, true, true, maxAttempts);
    }

    private void ensureJobExists(Scheduler scheduler) throws SchedulerException {
        if (!scheduler.checkExists(SendMyWarwickActivityJob.JOB_KEY)) {
            LOGGER.info("Creating job: " + SendMyWarwickActivityJob.JOB_KEY);
            scheduler.addJob(
                JobBuilder.newJob(SendMyWarwickActivityJob.class)
                    .withIdentity(SendMyWarwickActivityJob.JOB_KEY)
                    .storeDurably()
                    .requestRecovery()
                    .build(),
                true
            );
        }
    }

    private void queue(Activity activity, boolean isNotification, boolean isTransient, Scheduler scheduler) throws SchedulerException {
        ensureJobExists(scheduler);

        for (Instance instance : instances) {
            scheduler.scheduleJob(
                TriggerBuilder.newTrigger()
                    .forJob(SendMyWarwickActivityJob.JOB_KEY)
                    .usingJobData(SendMyWarwickActivityJob.INSTANCE_BASE_URL_DATA_KEY, instance.getBaseUrl())
                    .usingJobData(SendMyWarwickActivityJob.REQUEST_BODY_JOB_DATA_KEY, makeJsonBody(activity))
                    // JobDataMap values must be Strings when the 'useProperties' property is set.
                    .usingJobData(SendMyWarwickActivityJob.IS_NOTIFICATION_JOB_DATA_KEY, Boolean.toString(isNotification))
                    .usingJobData(SendMyWarwickActivityJob.IS_TRANSIENT_JOB_DATA_KEY, Boolean.toString(isTransient))
                    .usingJobData(SendMyWarwickActivityJob.CREATED_DATETIME_ISO8601_DATA_KEY, OffsetDateTime.now(DateTimeUtils.CLOCK_IMPLEMENTATION).toString())
                    .build()
            );
        }
    }

    @Override
    public void queueActivity(Activity activity, Scheduler scheduler) throws SchedulerException {
        queue(activity, false, false, scheduler);
    }

    @Override
    public void queueNotification(Activity activity, Scheduler scheduler) throws SchedulerException {
        queue(activity, true, false, scheduler);
    }

    @Override
    public void queueTransientPush(PushNotification pushNotification, Scheduler scheduler) throws SchedulerException {
        queue(pushNotification, true, true, scheduler);
    }

    @Override
    public CompletableFuture<List<Boolean>> hasPushRegistration(@NotNull String userId) {
        List<CompletableFuture<Boolean>> listOfCompletableFutures = instances.stream().map(instance -> {
            final HttpGet request = new HttpGet(instance.getHasPushRegistrationPath().replace(instance.getUsercodeReplaceString(), userId));
            CompletableFuture<Boolean> completableFuture = new CompletableFuture<>();

            request.addHeader(
                    "Authorization",
                    "Basic " + Base64.getEncoder().encodeToString((instance.getApiUser() + ":" + instance.getApiPassword()).getBytes(StandardCharsets.UTF_8)));
            request.addHeader(
                    "User-Agent",
                    "MyWarwickService");
            HasRegistrationHttpResponseCallback myWarwickHttpResponseCallback = new HasRegistrationHttpResponseCallback(
                    request.getURI().getPath(),
                    instance,
                    LOGGER,
                    completableFuture
            );

            RetryingHttpResponseCallback retryingHttpResponseCallback = new RetryingHttpResponseCallback(
                    httpclient,
                    request,
                    myWarwickHttpResponseCallback,
                    LOGGER,
                    3
            );

            httpclient.execute(request, retryingHttpResponseCallback);
            return completableFuture;
        }).collect(Collectors.toList());

        return CompletableFuture.allOf(listOfCompletableFutures.toArray(new CompletableFuture[0]))
                .thenApply(v -> listOfCompletableFutures
                        .stream()
                        .map(CompletableFuture::join)
                        .collect(Collectors.toList())
                );
    }

    String makeJsonBody(Activity activity) {
        String jsonString;
        try {
            jsonString = mapper.writeValueAsString(activity);
        } catch (JsonProcessingException e) {
            LOGGER.error(e.getMessage());
            jsonString = "{}";
        }
        return jsonString;
    }

    HttpPost makeRequest(String path, String json, String apiUser, String apiPassword, String providerId) {
        final HttpPost request = new HttpPost(path);
        request.addHeader(
                "Authorization",
                "Basic " + Base64.getEncoder().encodeToString((apiUser + ":" + apiPassword).getBytes(StandardCharsets.UTF_8)));
        request.addHeader(
                "Content-type",
                "application/json");
        request.addHeader(
                "User-Agent",
                "MyWarwickService/" + providerId);
        request.setEntity(new StringEntity(json, StandardCharsets.UTF_8));
        return request;
    }

    Set<Instance> getInstances() {
        return instances;
    }

    HttpClient getHttpClient() {
        return httpclient;
    }

    @PreDestroy
    public void destroy() throws Exception {
        httpclient.destroy();
    }
}
