package uk.ac.warwick.util.files.impl;

import java.io.File;
import java.io.InputStream;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.specialized.BlockBlobClient;

import org.jclouds.blobstore.BlobStore;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.domain.Blob;
import org.jclouds.blobstore.domain.BlobAccess;
import org.jclouds.blobstore.domain.BlobBuilder;
import org.jclouds.blobstore.domain.BlobMetadata;
import org.jclouds.blobstore.domain.ContainerAccess;
import org.jclouds.blobstore.domain.MultipartPart;
import org.jclouds.blobstore.domain.MultipartUpload;
import org.jclouds.blobstore.domain.PageSet;
import org.jclouds.blobstore.domain.StorageMetadata;
import org.jclouds.blobstore.options.CopyOptions;
import org.jclouds.blobstore.options.CreateContainerOptions;
import org.jclouds.blobstore.options.GetOptions;
import org.jclouds.blobstore.options.ListContainerOptions;
import org.jclouds.blobstore.options.PutOptions;
import org.jclouds.blobstore.strategy.internal.MultipartUploadSlicingAlgorithm;
import org.jclouds.domain.Location;
import org.jclouds.io.Payload;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WriteToAzureBlobStoreContext extends DelegatingBlobStoreContext {
  private static final Logger LOGGER = LoggerFactory.getLogger(WriteToAzureBlobStoreContext.class);
  
  private final BlobStoreContext baseContext;
  private final AzureReplicator replicator;

  public WriteToAzureBlobStoreContext(BlobStoreContext baseContext, BlobServiceClient azureClient) {
    super(baseContext);
    this.baseContext = baseContext;
    this.replicator = new AzureReplicator(baseContext, azureClient);
  }

  @Override
  public BlobStore getBlobStore() {
    return new WriteTwiceBlobStore(baseContext.getBlobStore());
  }

  public class WriteTwiceBlobStore implements BlobStore {
    private final BlobStore primary;

    public WriteTwiceBlobStore(BlobStore primary) {
      this.primary = primary;
    }

    @Override
    public BlobStoreContext getContext() {
      return WriteToAzureBlobStoreContext.this;
    }

    @Override
    public BlobBuilder blobBuilder(String name) {
      return primary.blobBuilder(name);
    }

    @Override
    public Set<? extends Location> listAssignableLocations() {
      return primary.listAssignableLocations();
    }

    @Override
    public PageSet<? extends StorageMetadata> list() {
      return primary.list();
    }

    @Override
    public boolean containerExists(String container) {
      boolean exists = primary.containerExists(container);
      catchingExceptions("checking container " + container, () -> {
        // What's going on, this method is just checking if the container exists,
        // but here we're creating it?
        // The client will only ask for it to be created if it doesn't exist in Swift,
        // resulting in missing Azure containers. Since we only ever call this immediately
        // before creating it, we can just create it here.
        replicator.ensureContainer(container);
        return null;
      });
      return exists;
    }

    @Override
    public boolean createContainerInLocation(Location location, String container) {
      boolean created = primary.createContainerInLocation(location, container);
      if (created) {
        catchingExceptions("creating container " + container, () -> {
          replicator.getAzureContainerClient(container).create();
          return null;
        });
      }
      return created;
    }

    @Override
    public boolean createContainerInLocation(Location location, String container, CreateContainerOptions options) {
      boolean created = primary.createContainerInLocation(location, container, options);
      if (created) {
        catchingExceptions("creating container " + container, () -> {
          replicator.getAzureContainerClient(container).create();
          return null;
        });
      }
      return created; 
    }

    @Override
    public ContainerAccess getContainerAccess(String container) {
      return primary.getContainerAccess(container);
    }

    @Override
    public void setContainerAccess(String container, ContainerAccess access) {
      primary.setContainerAccess(container, access);
    }

    @Override
    public PageSet<? extends StorageMetadata> list(String container) {
      return primary.list(container);
    }

    @Override
    public PageSet<? extends StorageMetadata> list(String container, ListContainerOptions options) {
      return primary.list(container, options);
    }

    @Override
    public void clearContainer(String container) {
      primary.clearContainer(container);
    }

    @Override
    public void clearContainer(String container, ListContainerOptions options) {
      primary.clearContainer(container, options);
    }

    @Override
    public void deleteContainer(String container) {
      primary.deleteContainer(container);
    }

    @Override
    public boolean deleteContainerIfEmpty(String container) {
      return primary.deleteContainerIfEmpty(container);
    }

    @Override
    public boolean directoryExists(String container, String directory) {
      return primary.directoryExists(container, directory);
    }

    @Override
    public void createDirectory(String container, String directory) {
      primary.createDirectory(container, directory);
    }

    @Override
    public void deleteDirectory(String containerName, String name) {
      primary.deleteDirectory(containerName, name);
    }

    @Override
    public boolean blobExists(String container, String name) {
      return primary.blobExists(container, name);
    }

    @Override
    public String putBlob(String container, Blob blob) {
      String result = primary.putBlob(container, blob);
      replicator.copyBlobToAzure(container, blob);
      return result;
    }

    @Override
    public String putBlob(String container, Blob blob, PutOptions options) {
      String result = primary.putBlob(container, blob, options);
      replicator.copyBlobToAzure(container, blob);
      return result;
    }

    @Override
    public String copyBlob(String fromContainer, String fromName, String toContainer, String toName, CopyOptions options) {
      String result = primary.copyBlob(fromContainer, fromName, toContainer, toName, options);
      // Could be problems with not passing the content length if it's actually a very large blob,
      // as it might not stage it into smaller blocks. I'm not sure we ever copy blobs in practice,
      // since we key on content hash so copies would use the same blob.
      replicator.copyBlobToAzure(toContainer, toName, null);
      return result;
    }

    @Override
    public BlobMetadata blobMetadata(String container, String name) {
      return primary.blobMetadata(container, name);
    }

    @Override
    public Blob getBlob(String container, String name) {
      return primary.getBlob(container, name);
    }

    @Override
    public Blob getBlob(String container, String name, GetOptions options) {
      return primary.getBlob(container, name, options);
    }

    @Override
    public void removeBlob(String container, String name) {
      primary.removeBlob(container, name);
    }

    @Override
    public void removeBlobs(String container, Iterable<String> names) {
      primary.removeBlobs(container, names);
    }

    @Override
    public BlobAccess getBlobAccess(String container, String name) {
      return primary.getBlobAccess(container, name);
    }

    @Override
    public void setBlobAccess(String container, String name, BlobAccess access) {
      primary.setBlobAccess(container, name, access);
    }

    @Override
    public long countBlobs(String container) {
      return primary.countBlobs(container);
    }

    @Override
    public long countBlobs(String container, ListContainerOptions options) {
      return primary.countBlobs(container, options);
    }

    @Override
    public MultipartUpload initiateMultipartUpload(String container, BlobMetadata blob, PutOptions options) {
      return primary.initiateMultipartUpload(container, blob, options);
    }

    @Override
    public void abortMultipartUpload(MultipartUpload mpu) {
      primary.abortMultipartUpload(mpu);
    }

    @Override
    public String completeMultipartUpload(MultipartUpload mpu, List<MultipartPart> parts) {
      String result = primary.completeMultipartUpload(mpu, parts);
      // mpu seems to have a null size metadata, but it's important to have it
      Long size = (mpu.blobMetadata() == null) ? null : mpu.blobMetadata().getSize();
      if (size == null) {
        BlobMetadata metadata = primary.blobMetadata(mpu.containerName(), mpu.blobName());
        size = metadata.getSize();
      }
      replicator.copyBlobToAzure(mpu.containerName(), mpu.blobName(), size);
      return result;
    }

    @Override
    public MultipartPart uploadMultipartPart(MultipartUpload mpu, int partNumber, Payload payload) {
      return primary.uploadMultipartPart(mpu, partNumber, payload);
    }

    @Override
    public List<MultipartPart> listMultipartUpload(MultipartUpload mpu) {
      return primary.listMultipartUpload(mpu);
    }

    @Override
    public List<MultipartUpload> listMultipartUploads(String container) {
      return primary.listMultipartUploads(container);
    }

    @Override
    public long getMinimumMultipartPartSize() {
      return primary.getMinimumMultipartPartSize();
    }

    @Override
    public long getMaximumMultipartPartSize() {
      return primary.getMaximumMultipartPartSize();
    }

    @Override
    public int getMaximumNumberOfParts() {
      return primary.getMaximumNumberOfParts();
    }

    @Override
    public void downloadBlob(String container, String name, File destination) {
      primary.downloadBlob(container, name, destination);
    }

    @Override
    public void downloadBlob(String container, String name, File destination, ExecutorService executor) {
      primary.downloadBlob(container, name, destination, executor);
    }

    @Override
    public InputStream streamBlob(String container, String name) {
      return primary.streamBlob(container, name);
    }

    @Override
    public InputStream streamBlob(String container, String name, ExecutorService executor) {
      return primary.streamBlob(container, name, executor);
    }
    
  }

  /**
   * Do some work while catching and logging any exceptions.
   */
  public static void catchingExceptions(String description, Callable<?> function) {
    try {
      function.call();
    } catch (Exception e) {
      LOGGER.error(String.format("Caught error in Azure operation - %s", description), e);
    }
  }
}
