/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.warwick.util.files.impl;

import com.azure.core.util.Context;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.specialized.BlockBlobClient;
import java.security.SecureRandom;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.domain.Blob;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.warwick.util.files.impl.RenameMode;
import uk.ac.warwick.util.files.impl.WriteToAzureBlobStoreContext;

class AzureReplicator {
    private final BlobStoreContext baseContext;
    private final BlobServiceClient azureClient;
    private final ExecutorService executor = Executors.newCachedThreadPool();
    private final RenameMode renameMode = RenameMode.NO_DOTS;
    private static final Duration ENSURE_CONTAINER_TIMEOUT = Duration.ofSeconds(3L);
    private static final long MAXIMUM_BLOCK_SIZE = 0x10000000L;
    private static final Logger LOGGER = LoggerFactory.getLogger(AzureReplicator.class);

    public AzureReplicator(BlobStoreContext baseContext, BlobServiceClient azureClient) {
        this.baseContext = baseContext;
        this.azureClient = azureClient;
    }

    public void copyBlobToAzure(String container, String blobName, Long blobSize) {
        boolean debug = LOGGER.isDebugEnabled();
        this.executor.submit(() -> WriteToAzureBlobStoreContext.catchingExceptions(String.format("copying %s/%s", container, blobName), () -> {
            String source = this.baseContext.getSigner().signGetBlob(container, blobName).getEndpoint().toString();
            BlobContainerClient containerClient = this.getAzureContainerClient(container);
            BlobClient blobClient = containerClient.getBlobClient(blobName);
            if (blobSize != null && blobSize >= 0x10000000L) {
                BlockBlobClient blockClient = blobClient.getBlockBlobClient();
                Blob swiftBlob = this.baseContext.getBlobStore().getBlob(container, blobName);
                blockClient.setMetadata(swiftBlob.getMetadata().getUserMetadata());
                List<BlobRange> sourceRanges = this.sliceRanges(blobSize, 0x10000000L);
                if (debug) {
                    LOGGER.debug("Uploading {}/{} in {} parts", new Object[]{container, blobName, sourceRanges.size()});
                }
                List blockIds = sourceRanges.stream().map(sourceRange -> {
                    String base64BlockId = this.newBlockIdBase64();
                    blockClient.stageBlockFromUrl(base64BlockId, source, sourceRange);
                    if (debug) {
                        LOGGER.debug("Staged {}/{} block ID {}", new Object[]{container, blobName, base64BlockId});
                    }
                    return base64BlockId;
                }).collect(Collectors.toList());
                blockClient.commitBlockList(blockIds);
                if (debug) {
                    LOGGER.debug("Committed {}/{}", (Object)container, (Object)blobName);
                }
            } else {
                if (debug) {
                    LOGGER.debug("Uploading {}/{}", (Object)container, (Object)blobName);
                }
                blobClient.copyFromUrl(source);
            }
            return null;
        }));
    }

    String newBlockIdBase64() {
        return Base64.getEncoder().encodeToString(this.newBlockId());
    }

    byte[] newBlockId() {
        SecureRandom ng = new SecureRandom();
        byte[] randomBytes = new byte[16];
        ng.nextBytes(randomBytes);
        return randomBytes;
    }

    List<BlobRange> sliceRanges(long length, long rangeSize) {
        ArrayList<BlobRange> ranges = new ArrayList<BlobRange>();
        for (long start = 0L; start < length; start += rangeSize) {
            long rangeLength = Math.min(rangeSize, length - start);
            ranges.add(new BlobRange(start, Long.valueOf(rangeLength)));
        }
        return ranges;
    }

    public void copyBlobToAzure(String container, Blob blob) {
        this.copyBlobToAzure(container, blob.getMetadata().getName(), blob.getMetadata().getSize());
    }

    public BlobContainerClient getAzureContainerClient(String containerName) {
        return this.azureClient.getBlobContainerClient(this.renameMode.rename(containerName));
    }

    public void ensureContainer(String container) {
        try {
            this.getAzureContainerClient(container).createWithResponse(null, null, ENSURE_CONTAINER_TIMEOUT, Context.NONE);
        }
        catch (BlobStorageException e) {
            if (Objects.equals(e.getErrorCode(), BlobErrorCode.CONTAINER_ALREADY_EXISTS)) {
                LOGGER.info("Azure container {} already exists", (Object)container);
            }
            throw e;
        }
    }
}

