package uk.ac.warwick.util.files.imageresize;

import com.google.common.io.ByteSource;
import com.google.common.io.Files;
import org.jmock.Expectations;
import org.jmock.Mockery;
import org.jmock.integration.junit4.JUnit4Mockery;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;
import org.springframework.util.FileCopyUtils;
import uk.ac.warwick.util.collections.Pair;
import uk.ac.warwick.util.files.DefaultFileStoreStatistics;
import uk.ac.warwick.util.files.FileReference;
import uk.ac.warwick.util.files.HashFileStore;
import uk.ac.warwick.util.files.hash.HashString;
import uk.ac.warwick.util.files.imageresize.ImageResizer.FileType;
import uk.ac.warwick.util.files.imageresize.ImageResizer.Orientation;
import uk.ac.warwick.util.files.impl.FileBackedHashFileReference;

import java.awt.image.BufferedImage;
import java.io.*;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import static org.junit.Assert.*;
import static uk.ac.warwick.util.files.imageresize.ImageIoResizer.getOrientation;

@SuppressWarnings("UnstableApiUsage")
@RunWith(Parameterized.class)
public class ImageIoResizerTest {
    ImageResizer resizer;

    ZonedDateTime lastModified = ZonedDateTime.now();

    private final Mockery m = new JUnit4Mockery();

    private final HashFileStore fileStore = m.mock(HashFileStore.class);

    final boolean outputToFiles = true;
    File temporaryDirectory;

    public ImageIoResizerTest(String name, ImageResizer imageResizer) {
        this.resizer = imageResizer;
    }

    @Parameters(name="{0}")
    public static List<Object[]> parameters() {
        return Arrays.asList(
                new Object[] { "ImageIoResizer", new ImageIoResizer() },
                new Object[] { "SingleThreadedImageResizer", new SingleThreadedImageResizer(new ImageIoResizer()) }
        );
    }

    @Before
    public void setup() {
        m.checking(new Expectations() {{
            allowing(fileStore).getStatistics(); will(returnValue(new DefaultFileStoreStatistics(fileStore)));
        }});
        if (outputToFiles) {
            String tmp = System.getProperty("java.io.tmpdir");
            temporaryDirectory = new File(tmp, "ImageIoResizerTest");
            temporaryDirectory.mkdirs();
        }
    }

    // Some gubbins so we can easily switch between output to files or to byte array
    interface Output {
        OutputStream getOutputStream() throws FileNotFoundException;
        InputStream read() throws FileNotFoundException;

        long size();
    }
    class FileOutput implements Output {
        private final FileOutputStream outputStream;
        private final File file;
        public FileOutput(File file) throws FileNotFoundException {
            this.outputStream = new FileOutputStream(file);
            this.file = file;
        }
        public OutputStream getOutputStream() {
            return outputStream;
        }
        public InputStream read() throws FileNotFoundException {
            return new FileInputStream(file);
        }
        public long size() {
            return file.length();
        }
    }
    class BytesOutput implements Output {
        private final ByteArrayOutputStream outputStream;
        public BytesOutput(ByteArrayOutputStream outputStream) {
            this.outputStream = outputStream;
        }
        public OutputStream getOutputStream() {
            return outputStream;
        }
        public InputStream read() throws FileNotFoundException {
            return new ByteArrayInputStream(outputStream.toByteArray());
        }
        public long size() {
            return outputStream.size();
        }
    }

    public Output newOutput(String name) throws IOException {
        if (outputToFiles) {
            File file = new File(temporaryDirectory, name);
            return new FileOutput(file);
        } else {
            return new BytesOutput(new ByteArrayOutputStream());
        }
    }

    /**
     * SBTWO-3903 a big zTXt chunk in a PNG highlighted a crappy decoder implementation in JAI -
     * it would complete but extremely slowly due to single-byte reading and String appends.
     * We no longer use JAI but the ImageIO codecs, and the TwelveMonkey ones as well, so they
     * shouldn't be affected but we keep this here to make sure.
     */
    @Test(timeout=5000)
    public void decodingZtxtChunks() throws Exception {
        ByteSource input = readResource("/bad-zTXt-chunk.png");
        Output output = newOutput("bad-zTXt-chunk.png");
        resizer.renderResized(input, null, null, output.getOutputStream(), 300, 300, FileType.jpg);
    }

    @Test
    public void resizeTallThinImage() throws IOException {
        // tallThinSample.jpg is 100 x 165 px
        ByteSource input = readResource("/tallThinSample.jpg");
        Output output = newOutput("tallThinSample_fromBytes.jpg");
        int maxWidth = 50;
        int maxHeight = 165;
        resizer.renderResized(input, null, null, output.getOutputStream(), maxWidth, maxHeight, FileType.jpg);

        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals(maxWidth, result.getWidth());
        assertTrue(maxHeight > result.getHeight());
    }

    @Test
    public void resizeAsFileNotByteArray() throws Exception {
        // tallThinSample.jpg is 100 x 165 px
        File input = new File(Objects.requireNonNull(this.getClass().getResource("/tallThinSample.jpg")).getFile());
        Output output = newOutput("tallThinSample_fromFile.jpg");
        int maxWidth = 50;
        int maxHeight = 165;
        resizer.renderResized(Files.asByteSource(input), null, null, output.getOutputStream(), maxWidth, maxHeight, FileType.jpg);

        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals(maxWidth, result.getWidth());
        assertTrue(maxHeight > result.getHeight());
    }

    /**
     * [SBTWO-1920] We didn't have a test that actually tested maxHeight worked;
     * and it didn't. Test that it does.
     */
    @Test
    public void resizeTallThinImageByHeight() throws IOException {
        // tallThinSample.jpg is 100 x 165 px
        ByteSource input = readResource("/tallThinSample.jpg");
        Output output = newOutput("tallThinSample_byHeight.jpg");
        int maxWidth = 100;
        int maxHeight = 155;
        resizer.renderResized(input, null, null, output.getOutputStream(), maxWidth, maxHeight, FileType.jpg);

        BufferedImage result = ImageReadUtils.read(output.read());

        assertEquals(155, result.getHeight());
    }

    @Test
    public void resizeShortWideImage() throws IOException {
        // shortWide.jpg is 200 x 150px
        ByteSource input = readResource("/shortWideSample.jpg");
        Output output = newOutput("shortWideSample.jpg");
        resizer.renderResized(input, null, null, output.getOutputStream(), 50, 165, FileType.jpg);

        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals(50, result.getWidth());
    }

    /**
     * SBTWO-3196 - we were only testing with byte arrays, meanwhile reading from a file started failing.
     */
    @Test
    public void fileReferenceInput() throws Exception {
        File f = new File(Objects.requireNonNull(this.getClass().getResource("/tallThinSample.jpg")).getFile());
        FileReference ref = new FileBackedHashFileReference(fileStore, f, new HashString("abcdef"));
        Output output = newOutput("tallThinSample_fileRef.jpg");
        resizer.renderResized(ref.asByteSource(), ref.getHash(), lastModified, output.getOutputStream(), 50, 165, FileType.jpg);
        byte[] bytes = FileCopyUtils.copyToByteArray(output.read());
        assertEquals(bytes.length, resizer.getResizedImageLength(ref.asByteSource(), ref.getHash(), lastModified, 50, 165, FileType.jpg));
    }

    @Test
    public void dontResizeLargerThanOriginal() throws IOException {
        // tallThinSample.jpg is 100 x 165 px
        ByteSource input = readResource("/tallThinSample.jpg");
        Output output = newOutput("tallThinSample_noEnlarge.jpg");
        resizer.renderResized(input, null, null, output.getOutputStream(), 150, 200, FileType.jpg);

        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals(100, result.getWidth());
        assertEquals(165, result.getHeight());

        assertEquals("Output should be the same as input", input.size(), output.size());
    }

    @Test
    public void PNGResizing() throws IOException {
        // award.png is 220x233px
        ByteSource input = readResource("/award.png");
        Output output = newOutput("award.jpg");
        resizer.renderResized(input, null, null, output.getOutputStream(), 110, 116, FileType.png);

        BufferedImage result = ImageReadUtils.read(output.read());

        // subsample average avoids black line at the bottom
        assertEquals(109, result.getWidth());
        assertEquals(116, result.getHeight());
    }

    @Test
    public void resizingWebPImageWillFunctionCorrectly() throws Exception {
        ByteSource input = readResource("/computer.webp");
        Output output = newOutput("computer.webp");
        resizer.renderResized(input, null, null, output.getOutputStream(), 1234, 0, FileType.webp);

        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals(1234, result.getWidth());
        assertEquals(822, result.getHeight());
    }

    @Test
    public void readWebpAsJpeg() throws Exception {
        ByteSource input = readResource("/computer.webp");
        Output output = newOutput("computer.jpg");
        // Someone has renamed their computer.webp to computer.jpg and uploaded it to the CMS,
        // which has blindly accepted that and is now asking it to be converted as a JPG. What will happen?
        resizer.renderResized(input, null, null, output.getOutputStream(), 300, 0, FileType.jpg);

        // Expecting an exception was logged and we fall back to outputting the original image -
        // though the calling app still thinks it's a JPG so who knows how that'll get served to the user.
        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals(5760, result.getWidth());
    }

    @Test
    public void GIFResizing() throws IOException {
        // award.png is 220x233px
        ByteSource input = readResource("/hugeimage.gif");
        Output output = newOutput("hugeimage.gif");
        resizer.renderResized(input, null, null, output.getOutputStream(), 110, 116, FileType.gif);

        BufferedImage result = ImageReadUtils.read(output.read());

        // subsample average avoids black line at the bottom
        assertEquals(110, result.getWidth());
        assertEquals(44, result.getHeight());
    }

    @Test
    public void resizeTheWorld() throws Exception {
        File input = new File(Objects.requireNonNull(this.getClass().getResource("/pendulum_crop1.jpg")).getFile());
        Output output = newOutput("pendulum_crop1.jpg");
        int maxWidth = 350;
        int maxHeight = 149;
        resizer.renderResized(Files.asByteSource(input), null, null, output.getOutputStream(), maxWidth, maxHeight, FileType.jpg);

        // Image resizer will refuse to resize.
        Pair<Integer, Integer> dimensions = ImageIoResizer.getDimensions(output.read());
        assertEquals(24803, dimensions.getLeft().intValue());
        assertEquals(10559, dimensions.getRight().intValue());
    }

    /**
     * Maybe there used to be something wrong with this image in the past, but it seems to resize fine now.
     * ImageIO may work around whatever is "bad" about it.
     */
    @Test
    public void badImage() throws Exception {

        File f = new File(Objects.requireNonNull(this.getClass().getResource("/October.jpg")).getFile());
        FileReference ref = new FileBackedHashFileReference(fileStore, f, new HashString("abcdef"));
        Output output = newOutput("October.jpg");
        resizer.renderResized(ref.asByteSource(), ref.getHash(), lastModified, output.getOutputStream(), 50, 165, FileType.jpg);

        assertEquals(output.size(), resizer.getResizedImageLength(ref.asByteSource(), ref.getHash(), lastModified, 50, 165, FileType.jpg));

        Pair<Integer, Integer> dimensions = ImageIoResizer.getDimensions(output.read());
        assertEquals(50, dimensions.getLeft().intValue());
        assertEquals(25, dimensions.getRight().intValue());
    }

    @Test
    public void orientation() throws Exception {
        assertEquals(Orientation.Normal, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_0.jpg"), FileType.jpg));
        assertEquals(Orientation.Normal, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_1.jpg"), FileType.jpg));
        assertEquals(Orientation.Mirrored, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_2.jpg"), FileType.jpg));
        assertEquals(Orientation.Rotate180, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_3.jpg"), FileType.jpg));
        assertEquals(Orientation.MirroredVertically, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_4.jpg"), FileType.jpg));
        assertEquals(Orientation.MirroredRotate270, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_5.jpg"), FileType.jpg));
        assertEquals(Orientation.Rotate90, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_6.jpg"), FileType.jpg));
        assertEquals(Orientation.MirroredRotate90, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_7.jpg"), FileType.jpg));
        assertEquals(Orientation.Rotate270, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_8.jpg"), FileType.jpg));

        for (int i = 0; i <= 8; i++) {
            ByteSource input = readResource("/orientation/Portrait_" + i + ".jpg");
            Output output = newOutput("Portrait_" + i + ".jpg");
            resizer.renderResized(input, null, null, output.getOutputStream(), 0, 0, FileType.jpg);

            BufferedImage result = ImageReadUtils.read(output.read());
            // This only really tests that rotation has happened - it won't spot any problems with mirroring
            assertEquals("Portrait "+i+" width", 1200, result.getWidth());
            assertEquals("Portrait "+i+" height", 1800, result.getHeight());
        }
    }

    @Test
    public void webpOrientation() throws Exception {
        assertEquals(Orientation.Normal, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_0.webp"), FileType.webp));
        assertEquals(Orientation.Normal, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_1.webp"), FileType.webp));
        assertEquals(Orientation.Mirrored, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_2.webp"), FileType.webp));
        assertEquals(Orientation.Rotate180, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_3.webp"), FileType.webp));
        assertEquals(Orientation.MirroredVertically, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_4.webp"), FileType.webp));
        assertEquals(Orientation.MirroredRotate270, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_5.webp"), FileType.webp));
        assertEquals(Orientation.Rotate90, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_6.webp"), FileType.webp));
        assertEquals(Orientation.MirroredRotate90, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_7.webp"), FileType.webp));
        assertEquals(Orientation.Rotate270, getOrientation(getClass().getResourceAsStream("/orientation/Portrait_8.webp"), FileType.webp));

        for (int i = 0; i <= 8; i++) {
            ByteSource input = readResource("/orientation/Portrait_" + i + ".webp");
            Output output = newOutput("Portrait_" + i + ".webp");
            resizer.renderResized(input, null, null, output.getOutputStream(), 0, 0, FileType.webp);

            BufferedImage result = ImageReadUtils.read(output.read());
            assertEquals(1200, result.getWidth());
            assertEquals(1800, result.getHeight());
        }
    }

    /**
     * Test what maxWidth means when we are also correcting EXIF 90 or 270 degree rotation -
     * is it the original width (which will become the height), or the new width after rotation?
     */
    @Test
    public void maxWidthAndOrientation() throws Exception {
        ByteSource portrait_6_90deg = readResource("/orientation/Portrait_6.webp");
        Output output = newOutput("Portrait_6_maxWidth.webp");
        resizer.renderResized(portrait_6_90deg, null, null, output.getOutputStream(), 1100, 0, FileType.webp);

        BufferedImage result = ImageReadUtils.read(output.read());
        assertEquals("maxWidth should apply to post-rotation width", 1100, result.getWidth());
    }

    private ByteSource readResource(String path) {
        return Files.asByteSource(new File(getClass().getResource(path).getFile()));
    }
}
