package uk.ac.warwick.util.files.impl;

import com.azure.core.util.Context;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.specialized.BlockBlobClient;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.domain.Blob;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.SecureRandom;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

import static uk.ac.warwick.util.files.impl.WriteToAzureBlobStoreContext.catchingExceptions;

/**
 * Sends requests to Azure to copy existing Swift blobs into Azure by
 * signing a Temporary URL and giving that to Azure to download directly. 
 */
class AzureReplicator {
  
  private final BlobStoreContext baseContext;
  private final BlobServiceClient azureClient;
  
  private final ExecutorService executor = Executors.newCachedThreadPool();

  public AzureReplicator(BlobStoreContext baseContext, BlobServiceClient azureClient) {
    this.baseContext = baseContext;
    this.azureClient = azureClient;
  }
  
  private final RenameMode renameMode = RenameMode.NO_DOTS;

  /**
   * Timeout for the request to create a container. We don't want to wait too long
   * if Azure is not responding as it delays startup too much.
   */
  private static final Duration ENSURE_CONTAINER_TIMEOUT = Duration.ofSeconds(3);

  private static final long MAXIMUM_BLOCK_SIZE = 256 * 1024 * 1024; // 256MB
  private static final Logger LOGGER = LoggerFactory.getLogger(AzureReplicator.class);

  public void copyBlobToAzure(String container, String blobName, Long blobSize) {
    boolean debug = LOGGER.isDebugEnabled();
    executor.submit(() -> {
      catchingExceptions(String.format("copying %s/%s", container, blobName), () -> {
        String source = baseContext.getSigner().signGetBlob(container, blobName).getEndpoint().toString();
        BlobContainerClient containerClient = getAzureContainerClient(container);
        BlobClient blobClient = containerClient.getBlobClient(blobName);
        if (blobSize != null && blobSize >= MAXIMUM_BLOCK_SIZE) {
          BlockBlobClient blockClient = blobClient.getBlockBlobClient();
          
          Blob swiftBlob = baseContext.getBlobStore().getBlob(container, blobName);
          blockClient.setMetadata(swiftBlob.getMetadata().getUserMetadata());

          List<BlobRange> sourceRanges = sliceRanges(blobSize, MAXIMUM_BLOCK_SIZE);

          if (debug) {
            LOGGER.debug("Uploading {}/{} in {} parts", container, blobName, sourceRanges.size());
          }

          List<String> blockIds = sourceRanges.stream()
            .map((sourceRange) -> {
              String base64BlockId = newBlockIdBase64();
              blockClient.stageBlockFromUrl(base64BlockId, source, sourceRange);
              if (debug) {
                LOGGER.debug("Staged {}/{} block ID {}", container, blobName, base64BlockId);
              }
              return base64BlockId;
            })
            .collect(Collectors.toList());

          blockClient.commitBlockList(blockIds);

          if (debug) {
            LOGGER.debug("Committed {}/{}", container, blobName);
          }
        } else {
          if (debug) {
            LOGGER.debug("Uploading {}/{}", container, blobName);
          }
          blobClient.copyFromUrl(source);
        }
        return null;
      });
    });
  }

  String newBlockIdBase64() {
    return Base64.getEncoder().encodeToString(newBlockId());
  }

  byte[] newBlockId() {
    SecureRandom ng = new SecureRandom();
    byte[] randomBytes = new byte[16];
    ng.nextBytes(randomBytes);
    return randomBytes;
  }

  List<BlobRange> sliceRanges(long length, long rangeSize) {
    List<BlobRange> ranges = new ArrayList<>();
    long start = 0;
    while (start < length) {
      long rangeLength = Math.min(rangeSize, length - start);
      ranges.add(new BlobRange(start, rangeLength));
      start += rangeSize;
    }
    return ranges;
  }

  public void copyBlobToAzure(String container, Blob blob) {
    copyBlobToAzure(container, blob.getMetadata().getName(), blob.getMetadata().getSize());
  }

  public BlobContainerClient getAzureContainerClient(String containerName) {
    return azureClient.getBlobContainerClient(renameMode.rename(containerName));
  }

  public void ensureContainer(String container) {
    try {
      getAzureContainerClient(container).createWithResponse(null, null, ENSURE_CONTAINER_TIMEOUT, Context.NONE);
    } catch (BlobStorageException e) {
      // We already catch this higher up, but this makes it less noisy - it's
      // not an error if the container already exists, it's good.
      if (Objects.equals(e.getErrorCode(), BlobErrorCode.CONTAINER_ALREADY_EXISTS)) {
        LOGGER.info("Azure container {} already exists", container);
      } else {
        throw e;
      }
    }
  }
}
