package uk.ac.warwick.util.files.imageresize;

import com.drew.imaging.ImageMetadataReader;
import com.drew.imaging.ImageProcessingException;
import com.drew.metadata.Metadata;
import com.drew.metadata.MetadataException;
import com.drew.metadata.exif.ExifDirectoryBase;
import com.drew.metadata.exif.ExifIFD0Directory;
import com.google.common.collect.Maps;
import com.google.common.io.ByteSource;
import com.twelvemonkeys.image.AffineTransformOp;
import com.twelvemonkeys.image.ResampleOp;
import org.apache.commons.imaging.ImageReadException;
import org.apache.commons.imaging.Imaging;
import org.apache.commons.imaging.common.ImageMetadata;
import org.apache.commons.imaging.formats.jpeg.JpegImageMetadata;
import org.apache.commons.imaging.formats.tiff.TiffField;
import org.apache.commons.imaging.formats.tiff.TiffImageMetadata;
import org.apache.commons.imaging.formats.tiff.constants.TiffTagConstants;
import org.imgscalr.Scalr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.warwick.util.collections.Pair;
import uk.ac.warwick.util.files.hash.HashString;

import javax.imageio.*;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.MemoryCacheImageOutputStream;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.ZonedDateTime;
import java.util.Map;
import java.util.NoSuchElementException;

public class ImageIoResizer implements ImageResizer {
    public static final int DEFAULT_MAX_WIDTH = 8000;
    public static final int DEFAULT_MAX_HEIGHT = 5000;

    private static final Logger LOGGER = LoggerFactory.getLogger(ImageIoResizer.class);

    private int maxWidthToResize = DEFAULT_MAX_WIDTH;
    private int maxHeightToResize = DEFAULT_MAX_HEIGHT;

    @Override
    public void renderResized(final ByteSource source, final HashString hash, final ZonedDateTime entityLastModified, final OutputStream out, final int maxWidth, final int maxHeight,
                              final FileType fileType) throws IOException {
        Pair<Integer, Integer> dimensions = getDimensions(source.openStream());
        if (isOversized(dimensions.getLeft(), dimensions.getRight())) {
            LOGGER.warn("Refusing to resize image of dimensions {}x{}: {}", dimensions.getLeft(), dimensions.getRight(), source);
            source.copyTo(out);
            return;
        }

        renderResizedStream(source, out, maxWidth, maxHeight, fileType, dimensions.getLeft(), dimensions.getRight());
    }

    private void renderResizedStream(final ByteSource in, final OutputStream out, final int maxWidth, final int maxHeight, final FileType fileType, float width, float height) throws IOException {
        long start = System.currentTimeMillis();

        final Orientation orientation;
        try (InputStream is = in.openStream()) {
            orientation = getOrientation(is, fileType);
        }

        if (!shouldResizeWidth(width, maxWidth) && !shouldResizeHeight(height, maxHeight) && orientation == Orientation.Normal) {
            // stream the input into the output
            in.copyTo(out);
            return;
        }

        try (ImageInputStream sourceSS = ImageIO.createImageInputStream(in.openStream())) {
            final ImageReader reader = getImageReader(fileType);
            reader.setInput(sourceSS);
            ImageReadParam params = reader.getDefaultReadParam();
            BufferedImage source = reader.read(0, params);

            Map<RenderingHints.Key, Object> map = Maps.newHashMap();
            map.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            map.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
            map.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
            map.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

            RenderingHints hints = new RenderingHints(map);

            BufferedImage transformedImage = transformImage(source, maxWidth, maxHeight, width, height, orientation, hints);

            // This may be null, so will want to write it below in that case
            ImageWriter writer = ImageIO.getImageWriter(reader);
            ImageWriteParam writeParam = null;

            switch (fileType) {
                case jpg:
                    if (writer == null) {
                        writer = ImageIO.getImageWritersByFormatName("JPEG").next();
                    }
                    writeParam = writer.getDefaultWriteParam();
                    writeParam.setCompressionMode(ImageWriteParam.MODE_COPY_FROM_METADATA);
                    break;
                case png:
                    if (writer == null) {
                        writer = ImageIO.getImageWritersByFormatName("PNG").next();
                    }
                    break;
                case webp:
                    if (writer == null) {
                        writer = ImageIO.getImageWritersByFormatName("WEBP").next();
                    }
                    break;
                default:
                    if (writer == null) {
                        throw new IllegalArgumentException("Unrecognised image");
                    }
            }

            assert writer != null;

            MemoryCacheImageOutputStream cacheImageOutputStream = new MemoryCacheImageOutputStream(out);
            writer.setOutput(cacheImageOutputStream);
            IIOImage outputImage = new IIOImage(transformedImage, /*thumbnails=*/null, null);
            writer.write(null, outputImage, writeParam);

            // SBTWO-10722 Flush the stream so data will be taken out of buffer and go straight to response output stream
            cacheImageOutputStream.flush();

            long duration = System.currentTimeMillis() - start;
            LOGGER.debug("MS to Resize image: {}", duration);
        } catch (Exception e) {
            LOGGER.error("Exception when resizing image, returning original image", e);
            in.copyTo(out);
        }
    }

    private static ImageReader getImageReader(FileType fileType) {
        final ImageReader reader;
        switch (fileType) {
            case gif:
                reader = ImageIO.getImageReadersByFormatName("GIF").next();
                break;
            case jpg:
                reader = ImageIO.getImageReadersByFormatName("JPEG").next();
                break;
            case png:
                reader = ImageIO.getImageReadersByFormatName("PNG").next();
                break;
            case webp:
                reader = ImageIO.getImageReadersByFormatName("WEBP").next();
                break;
            default:
                throw new IllegalArgumentException("Unrecognised image");
        }
        return reader;
    }

    /**
     * Reverse an EXIF orientation so that it can be displayed upright with no EXIF rotation in effect.
     * @param orientation current EXIF orientation
     * @return the AffineTransformOp to reverse this orientation
     */
    public static AffineTransformOp getInverseTransformOp(int width, int height, Orientation orientation) {
        AffineTransform tx = new AffineTransform();

        switch (orientation.rotationDegrees) {
            case 0: break; // no rotation
            case 90:
                tx.rotate(Math.PI / 2);
                tx.translate(0, -height);
                break;
            case 180:
                tx.rotate(Math.PI);
                tx.translate(-width, -height);
                break;
            case 270:
                tx.rotate(-Math.PI / 2);
                tx.translate(-width, 0);
                break;
            default:
                throw new IllegalArgumentException("Don't know how to handle rotationDegrees="+orientation.rotationDegrees);
        }

        if (orientation.mirrored) {
            tx.scale(-1, 1);
            tx.translate(-width, 0);
        }

        return new AffineTransformOp(tx, AffineTransformOp.TYPE_BILINEAR);
    }

    /**
     * Transforms an image to fit maxWidth and/or maxHeight,
     * and also applies any EXIF orientation (by doing the opposite
     * transform and not including EXIF orientation in the output).
     * <p>
     * Where an image is EXIF oriented 90 or 270 degrees, maxWidth and
     * maxHeight apply post-rotation so it should always be what you
     * would intuitively consider the width and the height of the picture.
     *
     * @param source an BufferedImage to scale
     * @param maxWidthIn maximum image width, if > 0
     * @param maxHeightIn maximum image height, if > 0
     * @param width width to calculate scale ratio
     * @param height height to calculate scale ratio
     * @param hints rendering hint
     * @return a scaled image
     */
    private BufferedImage transformImage(
            BufferedImage source,
            int maxWidthIn, int maxHeightIn,
            float width, float height,
            Orientation orientation,
            RenderingHints hints
    ) {
        int maxWidth = maxWidthIn;
        int maxHeight = maxHeightIn;
        if (orientation.rotationDegrees == 90 || orientation.rotationDegrees == 270) {
            // We're going to turn the image so width will become height
            //noinspection SuspiciousNameCombination
            maxWidth = maxHeightIn;
            //noinspection SuspiciousNameCombination
            maxHeight = maxWidthIn;
        }

        boolean tooWide = shouldResizeWidth(width, maxWidth);
        boolean tooHigh = shouldResizeHeight(height, maxHeight);
        if (tooWide || tooHigh || orientation != Orientation.Normal) {
            float scale = 1.0f;

            // if the image is too wide, scale down
            if (tooWide) {
                scale = maxWidth / width;
            }

            // if the image is too high, scale down
            // IF that makes it smaller than maxWidth has done already
            if (tooHigh) {
                float heightScale = maxHeight / height;
                if (heightScale < scale) {
                    scale = heightScale;
                }
            }

            // assume no resizing at first
            int targetWidth = (int) (width * scale);
            int targetHeight = (int) (height * scale);

            ResampleOp scaleOp = new ResampleOp(targetWidth, targetHeight, hints);
            AffineTransformOp orientationOp = getInverseTransformOp(targetWidth, targetHeight, orientation);

            return Scalr.apply(source, scaleOp, orientationOp);
        }

        return source;
    }

    @Override
    public long getResizedImageLength(ByteSource source, final HashString hash, final ZonedDateTime entityLastModified, int maxWidth, int maxHeight, FileType fileType) throws IOException {
        // Write image to a stream that discards all bytes, so we don't use more memory than necessary.
        LengthCountingOutputStream stream = new LengthCountingOutputStream();
        renderResized(source, hash, entityLastModified, stream, maxWidth, maxHeight, fileType);
        return stream.length();
    }

    private boolean shouldResizeWidth(final float width, final float maxWidth) {
        return width > maxWidth && maxWidth > 0;
    }

    private boolean shouldResizeHeight(final float height, final float maxHeight) {
        return height > maxHeight && maxHeight > 0;
    }

    public static Pair<Integer, Integer> getDimensions(InputStream input) throws IOException {
        // Optimisation: We don't actually need to read the whole image to get the width and height
        ImageInputStream imageStream = null;
        ImageReader reader = null;

        try {
            imageStream = ImageIO.createImageInputStream(input);

            // Safe to just call .next() as the NoSuchElementException will return null as desired
            reader = ImageIO.getImageReaders(imageStream).next();
            reader.setInput(imageStream, true, true);

            return Pair.of(reader.getWidth(0), reader.getHeight(0));
        } catch (NoSuchElementException e) {
            throw new NotAnImageException(e);
        } finally {
            if (reader != null)
                reader.dispose();

            if (imageStream != null)
                imageStream.close();

            input.close();
        }
    }

    public boolean isOversized(int width, int height) {
        // total number of pixels is more important than exact measurements
        return width * height > maxWidthToResize * maxHeightToResize;
    }

    public static Orientation getOrientation(InputStream input, FileType fileType) {
        // SBTWO-10722 Specifically handle WebP orientation
        if (fileType == FileType.webp) {
            // SBTWO-10722 Apache Commons Imaging library below does not support
            // WebP image, so we need to handle it separately here.
            return getOrientationForWebpFile(input);
        }

        try {
            ImageMetadata metadata = Imaging.getMetadata(input, "image." + fileType.name());
            if (metadata instanceof JpegImageMetadata) {
                TiffImageMetadata exif = ((JpegImageMetadata)metadata).getExif();
                if (exif != null) {
                    TiffField orientationField = exif.findField(TiffTagConstants.TIFF_TAG_ORIENTATION);
                    if (orientationField != null) {
                        return Orientation.fromExifOrientationValue(orientationField.getIntValue());
                    }
                }
            }
        } catch (ImageReadException | IOException e) {
            LOGGER.warn("Couldn't read the image orientation", e);
        }

        return Orientation.Normal;
    }

    /**
     * SBTWO-10722 Return the orientation from WebP input stream
     * @param inputStream WebP input stream
     * @return Orientation enum value
     */
    private static Orientation getOrientationForWebpFile(InputStream inputStream) {
        try {
            Metadata alternativeMetadata = ImageMetadataReader.readMetadata(inputStream);
            ExifIFD0Directory directory = alternativeMetadata.getFirstDirectoryOfType(ExifIFD0Directory.class);
            if (directory != null) {
                return Orientation.fromExifOrientationValue(directory.getInt(ExifDirectoryBase.TAG_ORIENTATION));
            }
        } catch (ImageProcessingException | IOException | MetadataException e) {
            LOGGER.warn("Couldn't read the image orientation", e);
        }

        return Orientation.Normal;
    }

    public void setMaxWidthToResize(int maxWidth) {
        this.maxWidthToResize = maxWidth;
    }

    public void setMaxHeightToResize(int maxHeight) {
        this.maxHeightToResize = maxHeight;
    }

    static class NotAnImageException extends IOException {

        private static final long serialVersionUID = -3845937464714363305L;

        NotAnImageException(Throwable t) {
            super(t);
        }
    }

}
