package uk.ac.warwick.util.convert.cloudconvert;

import com.cloudconvert.client.CloudConvertClient;
import com.cloudconvert.client.setttings.StringSettingsProvider;
import com.cloudconvert.dto.Status;
import com.cloudconvert.dto.request.ConvertFilesTaskRequest;
import com.cloudconvert.dto.request.UploadImportRequest;
import com.cloudconvert.dto.request.UrlExportRequest;
import com.cloudconvert.dto.response.JobResponse;
import com.cloudconvert.dto.response.TaskResponse;
import com.cloudconvert.exception.CloudConvertClientException;
import com.cloudconvert.exception.CloudConvertServerException;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import uk.ac.warwick.util.convert.DocumentConversionResult;
import uk.ac.warwick.util.convert.DocumentConversionService;
import uk.ac.warwick.util.convert.StreamByteSource;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;

public class CloudConvertDocumentConversionService implements DocumentConversionService, DisposableBean {

    private static final Logger LOGGER = LoggerFactory.getLogger(CloudConvertDocumentConversionService.class);
    
    private final CloudConvertClient client;

    public CloudConvertDocumentConversionService(String apiKey) {
        this(apiKey, false);
    }
    
    public CloudConvertDocumentConversionService(String apiKey, boolean sandbox) {
        try {
            // Second argument here is "webhook signing key", but we don't currently use webhooks
            // as we instead block on the job completing like in the olden days.
            this.client = new CloudConvertClient(new StringSettingsProvider(apiKey, "", sandbox));
        } catch (IOException e) {
            throw new IllegalStateException("Error initialising CloudConvertClient", e);
        }
    }

    /**
     * Uploads a file to CC, converts it, and exports it as a URL hosted by CC that expires after 24 hours.
     */
    @Override
    public CloudConvertDocumentConversionResult convert(ByteSource in, String filename, String inputFormat, String outputFormat) throws IOException {
        final String conversionId = UUID.randomUUID().toString();

        try {
            // Upload the file
            TaskResponse uploadResponse = requireNonNull(client.importUsing().upload(new UploadImportRequest(), in.openStream(), filename).getBody());
            
            final String convertTaskName = "convert-doc";
            final String exportTaskName = "export-doc";
            
            // Convert and export to URL as a single job
            JobResponse startResult = requireNonNull(client.jobs().create(ImmutableMap.of(
              convertTaskName, new ConvertFilesTaskRequest()
                .setInput(uploadResponse.getId())
                .setInputFormat(inputFormat)
                .setOutputFormat(outputFormat),
              exportTaskName, new UrlExportRequest()
                .setInput(convertTaskName)
            )).getBody());
            
            // Wait for the whole job to complete
            JobResponse finishResult = requireNonNull(client.jobs().wait(startResult.getId()).getBody());
            
            if (finishResult.getStatus() == Status.ERROR) {
                String message = finishResult.getTasks().stream().map(TaskResponse::getMessage).collect(Collectors.joining(","));
                throw new IOException(String.format("Conversion failed: [%s]", message));
            }
            
            TaskResponse exportResponse = finishResult.getTasks().stream()
              .filter(t -> t.getName().equals(exportTaskName))
              .findFirst()
              .orElseThrow(() -> new NullPointerException("Task "+exportTaskName+" not found"));
            List<Map<String, String>> files = exportResponse.getResult().getFiles();
            if (LOGGER.isDebugEnabled()) LOGGER.debug("Converted {} files: {}", files.size(), files);
            List<String> urls = files.stream()
              .map(f -> f.get("url"))
              .collect(Collectors.toList());
            
            return CloudConvertDocumentConversionResult.success(conversionId, urls);
        } catch (URISyntaxException | CloudConvertClientException | CloudConvertServerException e) {
            throw new IOException(e);
        }
    }

    @Override
    public ByteSource getConvertedFile(DocumentConversionResult result, String url) {
        // Use CC's internal HTTP client to download the file
        return StreamByteSource.from(() -> {
            try {
                return client.files().download(url).getBody();
            } catch (URISyntaxException | CloudConvertClientException | CloudConvertServerException e) {
                throw new IOException(e);
            }
        });
    }

    // Specific to CloudConvert - get the number of credits remaining for this month
    public int getRemainingCredits() throws IOException {
        try {
            return client.users().me().getBody().getCredits();
        } catch (URISyntaxException | CloudConvertClientException | CloudConvertServerException e) {
            throw new IOException(e);
        }
    }

    @Override
    public void destroy() throws Exception {
        client.close();
    }

}
