/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.warwick.util.cache;

import com.google.common.annotations.VisibleForTesting;
import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import uk.ac.warwick.util.cache.Cache;
import uk.ac.warwick.util.cache.CacheEntry;
import uk.ac.warwick.util.cache.CacheEntryFactoryWithDataInitialisation;
import uk.ac.warwick.util.cache.CacheEntryUpdateException;
import uk.ac.warwick.util.cache.CacheExpiryStrategy;
import uk.ac.warwick.util.cache.CacheListener;
import uk.ac.warwick.util.cache.CacheStatistics;
import uk.ac.warwick.util.cache.CacheStore;
import uk.ac.warwick.util.cache.CacheStoreUnavailableException;
import uk.ac.warwick.util.cache.CacheWithDataInitialisation;
import uk.ac.warwick.util.cache.TTLCacheExpiryStrategy;
import uk.ac.warwick.util.cache.UpdateCacheEntryTask;

public final class BasicCache<K extends Serializable, V extends Serializable, T>
implements CacheWithDataInitialisation<K, V, T> {
    private final CacheEntryFactoryWithDataInitialisation<K, V, T> entryFactory;
    private final List<CacheListener<K, V>> listeners = new ArrayList<CacheListener<K, V>>();
    private static ExecutorService staticThreadPool = new ThreadPoolExecutor(1, 16, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
    private ExecutorService threadPool = staticThreadPool;
    private final CacheStore<K, V> store;
    private CacheExpiryStrategy<K, V> expiryStrategy;
    private boolean asynchronousUpdateEnabled;
    private boolean asynchronousOnly;

    public BasicCache(@Nonnull CacheStore<K, V> cacheStore, @Nullable CacheEntryFactoryWithDataInitialisation<K, V, T> entryFactory, @Nonnull CacheExpiryStrategy<K, V> expiryStrategy, boolean asynchronousUpdateEnabled, boolean asynchronousOnly) {
        if (asynchronousOnly && !asynchronousUpdateEnabled) {
            throw new IllegalArgumentException("Can't have asynchronousOnly if no asynchronousUpdateEnabled");
        }
        this.entryFactory = entryFactory;
        this.expiryStrategy = expiryStrategy;
        this.store = cacheStore;
        this.asynchronousUpdateEnabled = asynchronousUpdateEnabled;
        this.asynchronousOnly = asynchronousOnly;
    }

    @Override
    public V get(K key) throws CacheEntryUpdateException {
        return this.get(key, null);
    }

    @Override
    public V get(K key, T data) throws CacheEntryUpdateException {
        return (V)((Serializable)this.getResult(key, data).getValue());
    }

    @Override
    public Map<K, V> get(List<K> keys) throws CacheEntryUpdateException {
        if (this.entryFactory == null) {
            throw new UnsupportedOperationException("Batch lookups only supported when a suitable EntryFactory is used");
        }
        if (!this.entryFactory.isSupportsMultiLookups()) {
            throw new UnsupportedOperationException("The given EntryFactory does not support batch lookups");
        }
        if (this.asynchronousOnly) {
            throw new UnsupportedOperationException("Multiple lookups not yet implemented for asynchronous-only caches.");
        }
        HashMap<K, V> results = new HashMap<K, V>();
        ArrayList<KeyEntry<K, V, T>> missing = new ArrayList<KeyEntry<K, V, T>>();
        ArrayList expired = new ArrayList();
        try {
            Map<Serializable, CacheEntry<Serializable, CacheEntry>> entries = this.store.getAll(keys);
            entries.forEach((key, entry) -> {
                if (entry == null || !this.asynchronousUpdateEnabled && this.isExpired((CacheEntry<K, V>)entry)) {
                    missing.add(new KeyEntry((Serializable)key, entry, null));
                } else {
                    results.put(key, entry.getValue());
                    if (this.isStale((CacheEntry<K, V>)entry)) {
                        expired.add(new KeyEntry((Serializable)key, entry, null));
                    }
                }
            });
        }
        catch (CacheStoreUnavailableException e) {
            missing.addAll(keys.stream().map(k -> new KeyEntry((Serializable)k, null, null)).collect(Collectors.toList()));
        }
        if (!missing.isEmpty()) {
            missing.addAll(expired);
            Map<K, CacheEntry<K, V>> updated = this.updateEntries(missing);
            for (Map.Entry<K, CacheEntry<K, V>> entry2 : updated.entrySet()) {
                results.put(entry2.getKey(), entry2.getValue().getValue());
            }
        } else if (!expired.isEmpty()) {
            this.threadPool.execute(UpdateCacheEntryTask.task(this, expired));
        }
        return results;
    }

    private CacheEntry<K, V> getOrNull(K key) {
        try {
            return this.store.get(key);
        }
        catch (CacheStoreUnavailableException e) {
            return null;
        }
    }

    @Override
    public Cache.Result<V> getResult(K key) throws CacheEntryUpdateException {
        return this.getResult(key, null);
    }

    @Override
    public Cache.Result<V> getResult(K key, T data) throws CacheEntryUpdateException {
        CacheEntry<K, V> entry = this.getOrNull(key);
        boolean expired = entry != null && this.isExpired(entry);
        boolean stale = entry != null && this.isStale(entry);
        boolean updating = false;
        if (entry != null && !expired) {
            this.broadcastHit(key, entry);
            if (stale && !this.asynchronousOnly) {
                this.threadPool.execute(UpdateCacheEntryTask.task(this, new KeyEntry<K, V, T>(key, entry, data)));
                updating = true;
            }
        } else if (this.entryFactory != null) {
            if (!(this.asynchronousOnly || entry != null && this.asynchronousUpdateEnabled)) {
                entry = this.updateEntry(new KeyEntry<K, V, T>(key, entry, data));
            } else {
                this.threadPool.execute(UpdateCacheEntryTask.task(this, new KeyEntry<K, V, T>(key, entry, data)));
                updating = true;
            }
        }
        Object value = null;
        long lastUpdated = -1L;
        if (entry != null) {
            value = entry.getValue();
            lastUpdated = entry.getTimestamp();
            updating = updating || entry.isUpdating();
        }
        return new Cache.ResultImpl<Object>(value, updating, lastUpdated);
    }

    private void broadcastMiss(K key, CacheEntry<K, V> newEntry) {
        for (CacheListener<K, V> listener : this.listeners) {
            listener.cacheMiss(key, newEntry);
        }
    }

    private void broadcastHit(K key, CacheEntry<K, V> entry) {
        for (CacheListener<K, V> listener : this.listeners) {
            listener.cacheHit(key, entry);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CacheEntry<K, V> updateEntry(KeyEntry<K, V, T> kEntry) throws CacheEntryUpdateException {
        Object key = kEntry.key;
        Object data = kEntry.data;
        CacheEntry foundEntry = kEntry.entry;
        CacheEntry entry = this.getOrNull(key);
        if (entry == null || entry.equals(foundEntry) && !entry.isUpdating()) {
            if (entry != null) {
                entry.setUpdating(true);
            }
            try {
                V newValue = this.entryFactory.create(key, data);
                entry = this.newEntry(key, newValue);
                if (this.entryFactory.shouldBeCached(newValue)) {
                    this.put(entry);
                }
                this.broadcastMiss(key, entry);
            }
            finally {
                if (entry != null) {
                    entry.setUpdating(false);
                }
            }
        } else {
            this.broadcastHit(key, entry);
        }
        return entry;
    }

    Map<K, CacheEntry<K, V>> updateEntries(Collection<KeyEntry<K, V, T>> kentries) throws CacheEntryUpdateException {
        HashMap<Serializable, CacheEntry<Serializable, Serializable>> result = new HashMap<Serializable, CacheEntry<Serializable, Serializable>>();
        ArrayList keys = new ArrayList();
        for (KeyEntry<K, V, T> kentry : kentries) {
            CacheEntry foundEntry = kentry.entry;
            if (foundEntry != null) {
                foundEntry.setUpdating(true);
            }
            keys.add(kentry.key);
        }
        V createdMap = this.entryFactory.create(keys);
        for (Map.Entry created : createdMap.entrySet()) {
            Serializable key = (Serializable)created.getKey();
            Serializable value = (Serializable)created.getValue();
            CacheEntry<Serializable, Serializable> entry = new CacheEntry<Serializable, Serializable>(key, value);
            if (this.entryFactory.shouldBeCached(value)) {
                this.put(entry);
            }
            result.put(key, entry);
            this.broadcastMiss(key, entry);
        }
        return result;
    }

    @Override
    public void put(CacheEntry<K, V> entry) {
        try {
            this.store.put(entry, this.expiryStrategy.getTTLDuration(entry));
        }
        catch (CacheStoreUnavailableException cacheStoreUnavailableException) {
            // empty catch block
        }
    }

    @Override
    public boolean remove(K key) {
        try {
            return this.store.remove(key);
        }
        catch (CacheStoreUnavailableException e) {
            return false;
        }
    }

    private CacheEntry<K, V> newEntry(K key, V newValue) {
        return new CacheEntry<K, V>(key, newValue);
    }

    private boolean isExpired(CacheEntry<K, V> entry) {
        return this.expiryStrategy.isExpired(entry);
    }

    private boolean isStale(CacheEntry<K, V> entry) {
        return this.expiryStrategy.isStale(entry);
    }

    @Override
    public void addCacheListener(CacheListener<K, V> listener) {
        this.listeners.add(listener);
    }

    @Override
    public CacheStatistics getStatistics() throws CacheStoreUnavailableException {
        return this.store.getStatistics();
    }

    public boolean isAsynchronousUpdateEnabled() {
        return this.asynchronousUpdateEnabled;
    }

    public boolean isAsynchronousOnly() {
        return this.asynchronousOnly;
    }

    @Override
    public boolean clear() {
        try {
            return this.store.clear();
        }
        catch (CacheStoreUnavailableException e) {
            return false;
        }
    }

    @Override
    public boolean contains(K key) {
        try {
            return this.store.contains(key);
        }
        catch (CacheStoreUnavailableException e) {
            return false;
        }
    }

    @Override
    public String getName() {
        return this.store.getName();
    }

    @Override
    public void shutdown() {
        this.store.shutdown();
    }

    @Override
    public void setMaxSize(int cacheSize) {
        this.store.setMaxSize(cacheSize);
    }

    @Override
    public void setTimeout(int seconds) {
        this.setExpiryStrategy(TTLCacheExpiryStrategy.forTTL(Duration.ofSeconds(seconds)));
    }

    @Override
    public void setExpiryStrategy(CacheExpiryStrategy<K, V> expiryStrategy) {
        this.expiryStrategy = expiryStrategy;
    }

    @Override
    public void setAsynchronousUpdateEnabled(boolean asynchronousUpdateEnabled) {
        this.asynchronousUpdateEnabled = asynchronousUpdateEnabled;
    }

    @Deprecated
    public void setAsynchronousOnly(boolean asynchronousOnly) {
        this.asynchronousOnly = asynchronousOnly;
    }

    public static void setThreadPool(ExecutorService threadPool) {
        if (staticThreadPool != null) {
            staticThreadPool.shutdown();
        }
        staticThreadPool = threadPool;
    }

    public final void setLocalThreadPool(ExecutorService newThreadPool) {
        this.threadPool = newThreadPool;
    }

    @VisibleForTesting
    CacheStore<K, V> getCacheStore() {
        return this.store;
    }

    static class KeyEntry<K extends Serializable, V extends Serializable, T> {
        public final K key;
        public final T data;
        public final CacheEntry<K, V> entry;

        public KeyEntry(K k, CacheEntry<K, V> e, T d) {
            this.key = k;
            this.entry = e;
            this.data = d;
        }
    }
}

