/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.warwick.util.cache;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import org.jmock.lib.concurrent.DeterministicScheduler;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import uk.ac.warwick.util.cache.BasicCache;
import uk.ac.warwick.util.cache.Cache;
import uk.ac.warwick.util.cache.CacheEntry;
import uk.ac.warwick.util.cache.CacheEntryFactory;
import uk.ac.warwick.util.cache.CacheEntryUpdateException;
import uk.ac.warwick.util.cache.CacheExpiryStrategy;
import uk.ac.warwick.util.cache.CacheStatistics;
import uk.ac.warwick.util.cache.CacheStore;
import uk.ac.warwick.util.cache.CacheStoreUnavailableException;
import uk.ac.warwick.util.cache.CacheWithDataInitialisation;
import uk.ac.warwick.util.cache.Caches;
import uk.ac.warwick.util.cache.SingularCacheEntryFactory;
import uk.ac.warwick.util.cache.TTLCacheExpiryStrategy;
import uk.ac.warwick.util.cache.caffeine.CaffeineCacheStore;

public class BasicCacheTest {
    private static final String CACHE_NAME = "customCache";
    private static final String SLOW_CACHE_NAME = "slowCustomCache";
    private static final String NO_FACTORY_CACHE_NAME = "noFactoryCustomCache";
    private Cache<String, String> cache;
    private Cache<String, String> slowCache;
    private BrokenCacheEntryFactory slowFactory;
    private Cache<String, String> noFactoryCache;
    private final CacheExpiryStrategy<String, String> shortExpiry = TTLCacheExpiryStrategy.forTTL((Duration)Duration.ofMillis(100L));

    @Before
    public void setUp() {
        this.cache = Caches.builder((String)CACHE_NAME, (CacheEntryFactory)new SingularCacheEntryFactory<String, String>(){

            public String create(String key) {
                return "Value for " + key;
            }

            public boolean shouldBeCached(String val) {
                return true;
            }
        }).expireAfterWrite(Duration.ofSeconds(100L)).maximumSize(100L).build();
        this.slowFactory = new BrokenCacheEntryFactory();
        this.slowCache = Caches.builder((String)SLOW_CACHE_NAME, (CacheEntryFactory)this.slowFactory).expireAfterWrite(Duration.ofSeconds(100L)).maximumSize(100L).build();
        this.noFactoryCache = Caches.builder((String)NO_FACTORY_CACHE_NAME).expireAfterWrite(Duration.ofSeconds(100L)).maximumSize(100L).build();
    }

    @After
    public void tearDown() {
        this.cache.shutdown();
        this.slowCache.shutdown();
        this.noFactoryCache.shutdown();
    }

    @Test
    public void getMissingValue() throws Exception {
        Assert.assertEquals((Object)"Value for dog", (Object)this.cache.get((Serializable)((Object)"dog")));
        Assert.assertEquals((Object)"Value for cat", (Object)this.cache.get((Serializable)((Object)"cat")));
        Assert.assertSame((Object)this.cache.get((Serializable)((Object)"frog")), (Object)this.cache.get((Serializable)((Object)"frog")));
    }

    @Test
    public void noFactory() throws Exception {
        this.noFactoryCache.put(new CacheEntry((Serializable)((Object)"cat"), (Serializable)((Object)"meow")));
        Assert.assertNull((Object)this.noFactoryCache.get((Serializable)((Object)"dog")));
        Assert.assertEquals((Object)"meow", (Object)this.noFactoryCache.get((Serializable)((Object)"cat")));
    }

    @Test
    public void multiLookupsSynchronous() throws Exception {
        this.slowFactory.stopBlocking();
        HashMap<String, String> expected = new HashMap<String, String>();
        expected.put("dog", "Value for dog");
        expected.put("cat", "Value for cat");
        Assert.assertEquals(expected, (Object)this.slowCache.get(Arrays.asList("dog", "cat")));
    }

    @Test(expected=UnsupportedOperationException.class)
    public void multiLookupsAsynchronousOnly() throws Exception {
        this.slowFactory.stopBlocking();
        this.slowCache = Caches.builder((String)"customSlowCache1", (CacheEntryFactory)this.slowFactory).expireAfterWrite(Duration.ofSeconds(100L)).asynchronousOnly().maximumSize(100L).build();
        this.slowCache.get(Arrays.asList("dog", "cat"));
    }

    @Test
    public void slowConcurrentLookups() throws Exception {
        this.assertFactoryCount(0);
        Runnable getDog = () -> {
            try {
                this.slowCache.get((Serializable)((Object)"dog"));
            }
            catch (CacheEntryUpdateException e) {
                throw e.getRuntimeException();
            }
        };
        Thread t1 = new Thread(getDog);
        Thread t2 = new Thread(getDog);
        t1.start();
        t2.start();
        Thread.sleep(100L);
        this.slowFactory.addFastRequest("frog");
        Assert.assertEquals((Object)"Value for frog", (Object)this.slowCache.get((Serializable)((Object)"frog")));
        this.slowFactory.stopBlocking();
        t1.join();
        t2.join();
        List<String> requests = this.slowFactory.getObjectsCreated();
        Assert.assertEquals((long)3L, (long)requests.size());
        Assert.assertEquals((Object)"frog", (Object)requests.get(0));
        Assert.assertEquals((Object)"dog", (Object)requests.get(1));
    }

    @Test
    public void asynchronousUpdates() throws Exception {
        this.slowCache = Caches.builder((String)"asynchronousUpdates", (CacheEntryFactory)this.slowFactory).expiryStategy(this.shortExpiry).asynchronous().maximumSize(100L).build();
        this.slowFactory.addFastRequest("one");
        this.assertFactoryCount(0);
        String result1 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        String result2 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        Thread.sleep(150L);
        this.assertFactoryCount(1);
        String result3 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        this.assertFactoryCount(1);
        Thread.sleep(50L);
        this.assertFactoryCount(2);
        String result4 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        this.assertFactoryCount(2);
        Assert.assertSame((String)"Should have got cached value the second time", (Object)result1, (Object)result2);
        Assert.assertSame((String)"Should have got stale value", (Object)result2, (Object)result3);
        Assert.assertNotSame((String)"Should have returned async-updated result", (Object)result3, (Object)result4);
        this.slowCache.shutdown();
    }

    private void assertFactoryCount(int number) {
        Assert.assertEquals((long)number, (long)this.slowFactory.getObjectsCreated().size());
    }

    @Test
    public void sizeRestriction() throws Exception {
        int cacheSize = 4;
        this.cache = Caches.builder((String)"sizeRestriction", (CacheEntryFactory)new SingularCacheEntryFactory<String, String>(){

            public String create(String key) {
                return "Value for " + key;
            }

            public boolean shouldBeCached(String val) {
                return true;
            }
        }).expireAfterWrite(Duration.ofSeconds(100L)).maximumSize((long)cacheSize).build();
        Assert.assertEquals((String)"Should start empty", (long)0L, (long)this.cache.getStatistics().getCacheSize());
        this.cache.get((Serializable)((Object)"one"));
        this.cache.get((Serializable)((Object)"two"));
        this.cache.get((Serializable)((Object)"three"));
        this.cache.get((Serializable)((Object)"four"));
        Assert.assertEquals((long)4L, (long)this.cache.getStatistics().getCacheSize());
        this.cache.get((Serializable)((Object)"five"));
        this.cache.get((Serializable)((Object)"six"));
        ((CaffeineCacheStore)((BasicCache)this.cache).getCacheStore()).getCaffeineCache().cleanUp();
        Assert.assertEquals((long)4L, (long)this.cache.getStatistics().getCacheSize());
        Assert.assertEquals((String)"Shouldn't exceed maximum size", (long)cacheSize, (long)this.cache.getStatistics().getCacheSize());
        Assert.assertFalse((String)"Oldest entry should be evicted", (boolean)this.cache.contains((Serializable)((Object)"one")));
    }

    @Test
    public void expiry() throws Exception {
        this.slowCache = Caches.builder((String)"expiry", (CacheEntryFactory)this.slowFactory).expiryStategy(this.shortExpiry).maximumSize(100L).build();
        this.slowFactory.addFastRequest("one");
        String result1 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        System.err.println("Got first item");
        String result2 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        Thread.sleep(150L);
        String result3 = (String)((Object)this.slowCache.get((Serializable)((Object)"one")));
        Assert.assertSame((Object)result1, (Object)result2);
        Assert.assertNotSame((Object)result1, (Object)result3);
        this.slowCache.shutdown();
    }

    @Test(expected=IllegalStateException.class)
    public void unboundedCaffeineCachesDisallowed() throws Exception {
        CacheWithDataInitialisation cache = Caches.builder((String)"failingCache", (CacheEntryFactory)this.slowFactory, (Caches.CacheStrategy)Caches.CacheStrategy.CaffeineRequired).expiryStategy(this.shortExpiry).build();
        cache.shutdown();
    }

    @Test
    public void concurrentInitialRequests() throws Exception {
        CacheStore<String, String> store = new CacheStore<String, String>(){
            private final Set<String> called = new HashSet<String>();

            public CacheEntry<String, String> get(String key) {
                if (this.called.contains(key)) {
                    return new CacheEntry((Serializable)((Object)key), (Serializable)((Object)("Value for " + key)));
                }
                this.called.add(key);
                return null;
            }

            public Map<String, CacheEntry<String, String>> getAll(List<String> keys) throws CacheStoreUnavailableException {
                return keys.stream().collect(HashMap::new, (m, k) -> this.get((String)k), HashMap::putAll);
            }

            public void put(CacheEntry<String, String> entry, Duration ttl) {
                this.called.add((String)((Object)entry.getKey()));
            }

            public boolean remove(String key) {
                throw new UnsupportedOperationException();
            }

            public CacheStatistics getStatistics() {
                throw new UnsupportedOperationException();
            }

            public void setMaxSize(int size) {
                throw new UnsupportedOperationException();
            }

            public boolean clear() {
                throw new UnsupportedOperationException();
            }

            public boolean contains(String key) {
                throw new UnsupportedOperationException();
            }

            public void shutdown() {
                throw new UnsupportedOperationException();
            }

            public String getName() {
                throw new UnsupportedOperationException();
            }
        };
        BasicCache cache = new BasicCache((CacheStore)store, Caches.wrapFactoryWithoutDataInitialisation((CacheEntryFactory)new SingularCacheEntryFactory<String, String>(){

            public String create(String key) {
                return "Value for " + key;
            }

            public boolean shouldBeCached(String val) {
                return true;
            }
        }), (CacheExpiryStrategy)TTLCacheExpiryStrategy.forTTL((Duration)Duration.ofSeconds(100L)), false, false);
        Assert.assertEquals((Object)"Value for steve", (Object)cache.get((Serializable)((Object)"steve")));
    }

    @Test
    public void asynchronousOnlyGetReturnsNull() throws Exception {
        DeterministicScheduler scheduler = new DeterministicScheduler();
        BasicCache cache = new BasicCache(Caches.builder((String)CACHE_NAME).expireAfterWrite(Duration.ofSeconds(100L)).maximumSize(100L).buildStore(), Caches.wrapFactoryWithoutDataInitialisation((CacheEntryFactory)new SingularCacheEntryFactory<String, String>(){

            public String create(String key) {
                return "Value for " + key;
            }

            public boolean shouldBeCached(String val) {
                return true;
            }
        }), (CacheExpiryStrategy)TTLCacheExpiryStrategy.forTTL((Duration)Duration.ofSeconds(100L)), true, true);
        cache.setLocalThreadPool((ExecutorService)scheduler);
        Assert.assertNull((String)"Should be null", (Object)cache.get((Serializable)((Object)"alan")));
        scheduler.runUntilIdle();
        Assert.assertEquals((String)"Should be set", (Object)"Value for alan", (Object)cache.get((Serializable)((Object)"alan")));
    }

    @Test
    public void asynchronousOnlyGetResultReturnsNull() throws Exception {
        DeterministicScheduler scheduler = new DeterministicScheduler();
        BasicCache cache = new BasicCache(Caches.builder((String)CACHE_NAME).expireAfterWrite(Duration.ofSeconds(100L)).maximumSize(100L).buildStore(), Caches.wrapFactoryWithoutDataInitialisation((CacheEntryFactory)new SingularCacheEntryFactory<String, String>(){

            public String create(String key) {
                return "Value for " + key;
            }

            public boolean shouldBeCached(String val) {
                return true;
            }
        }), (CacheExpiryStrategy)TTLCacheExpiryStrategy.forTTL((Duration)Duration.ofSeconds(100L)), true, true);
        cache.setLocalThreadPool((ExecutorService)scheduler);
        Cache.Result result = cache.getResult((Serializable)((Object)"alan"));
        Assert.assertEquals((long)-1L, (long)result.getLastUpdated());
        Assert.assertTrue((String)"Should be updating", (boolean)result.isUpdating());
        Assert.assertNull((String)"Should be null", (Object)result.getValue());
        scheduler.runUntilIdle();
        result = cache.getResult((Serializable)((Object)"alan"));
        Assert.assertTrue((String)"Should have recent timestamp", (result.getLastUpdated() + 1000L > System.currentTimeMillis() ? 1 : 0) != 0);
        Assert.assertFalse((String)"Should not be updating", (boolean)result.isUpdating());
        Assert.assertEquals((Object)"Value for alan", (Object)result.getValue());
    }

    class BrokenCacheEntryFactory
    implements CacheEntryFactory<String, String> {
        private volatile boolean blocking = true;
        private List<String> requests = Collections.synchronizedList(new ArrayList());
        private Set<String> fastRequests = new HashSet<String>();

        BrokenCacheEntryFactory() {
        }

        public synchronized String create(String key) {
            if (!this.fastRequests.contains(key)) {
                while (this.blocking) {
                    try {
                        this.wait();
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
            this.requests.add(key);
            return "Value for " + key;
        }

        synchronized void stopBlocking() {
            this.blocking = false;
            this.notifyAll();
        }

        List<String> getObjectsCreated() {
            return this.requests;
        }

        void addFastRequest(String s) {
            this.fastRequests.add(s);
        }

        public Map<String, String> create(List<String> keys) {
            HashMap<String, String> response = new HashMap<String, String>();
            for (String key : keys) {
                response.put(key, this.create(key));
            }
            return response;
        }

        public boolean isSupportsMultiLookups() {
            return true;
        }

        public boolean shouldBeCached(String val) {
            return true;
        }
    }
}

