/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.warwick.util.files.imageresize;

import com.drew.imaging.ImageMetadataReader;
import com.drew.imaging.ImageProcessingException;
import com.drew.metadata.Metadata;
import com.drew.metadata.MetadataException;
import com.drew.metadata.exif.ExifIFD0Directory;
import com.google.common.collect.Maps;
import com.google.common.io.ByteSource;
import com.twelvemonkeys.image.AffineTransformOp;
import com.twelvemonkeys.image.ResampleOp;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.NoSuchElementException;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.MemoryCacheImageOutputStream;
import org.apache.commons.imaging.ImageReadException;
import org.apache.commons.imaging.Imaging;
import org.apache.commons.imaging.common.ImageMetadata;
import org.apache.commons.imaging.formats.jpeg.JpegImageMetadata;
import org.apache.commons.imaging.formats.tiff.TiffField;
import org.apache.commons.imaging.formats.tiff.TiffImageMetadata;
import org.apache.commons.imaging.formats.tiff.constants.TiffTagConstants;
import org.apache.commons.imaging.formats.tiff.taginfos.TagInfo;
import org.imgscalr.Scalr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.warwick.util.collections.Pair;
import uk.ac.warwick.util.files.hash.HashString;
import uk.ac.warwick.util.files.imageresize.ImageResizer;
import uk.ac.warwick.util.files.imageresize.LengthCountingOutputStream;

public class ImageIoResizer
implements ImageResizer {
    public static final int DEFAULT_MAX_WIDTH = 8000;
    public static final int DEFAULT_MAX_HEIGHT = 5000;
    private static final Logger LOGGER = LoggerFactory.getLogger(ImageIoResizer.class);
    private int maxWidthToResize = 8000;
    private int maxHeightToResize = 5000;

    @Override
    public void renderResized(ByteSource source, HashString hash, ZonedDateTime entityLastModified, OutputStream out, int maxWidth, int maxHeight, ImageResizer.FileType fileType) throws IOException {
        Pair<Integer, Integer> dimensions = ImageIoResizer.getDimensions(source.openStream());
        if (this.isOversized(dimensions.getLeft(), dimensions.getRight())) {
            LOGGER.warn("Refusing to resize image of dimensions {}x{}: {}", new Object[]{dimensions.getLeft(), dimensions.getRight(), source});
            source.copyTo(out);
            return;
        }
        this.renderResizedStream(source, out, maxWidth, maxHeight, fileType, dimensions.getLeft().intValue(), dimensions.getRight().intValue());
    }

    private void renderResizedStream(ByteSource in, OutputStream out, int maxWidth, int maxHeight, ImageResizer.FileType fileType, float width, float height) throws IOException {
        ImageResizer.Orientation orientation;
        long start = System.currentTimeMillis();
        try (InputStream is = in.openStream();){
            orientation = ImageIoResizer.getOrientation(is, fileType);
        }
        if (!this.shouldResizeWidth(width, maxWidth) && !this.shouldResizeHeight(height, maxHeight) && orientation == ImageResizer.Orientation.Normal) {
            in.copyTo(out);
            return;
        }
        try {
            var12_11 = null;
            try (ImageInputStream sourceSS = ImageIO.createImageInputStream(in.openStream());){
                ImageReader reader = ImageIoResizer.getImageReader(fileType);
                reader.setInput(sourceSS);
                ImageReadParam params = reader.getDefaultReadParam();
                BufferedImage source = reader.read(0, params);
                HashMap map = Maps.newHashMap();
                map.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
                map.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
                map.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
                map.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
                RenderingHints hints = new RenderingHints(map);
                BufferedImage transformedImage = this.transformImage(source, maxWidth, maxHeight, width, height, orientation, hints);
                ImageWriter writer = ImageIO.getImageWriter(reader);
                ImageWriteParam writeParam = null;
                switch (fileType) {
                    case jpg: {
                        if (writer == null) {
                            writer = ImageIO.getImageWritersByFormatName("JPEG").next();
                        }
                        writeParam = writer.getDefaultWriteParam();
                        writeParam.setCompressionMode(3);
                        break;
                    }
                    case png: {
                        if (writer != null) break;
                        writer = ImageIO.getImageWritersByFormatName("PNG").next();
                        break;
                    }
                    case webp: {
                        if (writer != null) break;
                        writer = ImageIO.getImageWritersByFormatName("WEBP").next();
                        break;
                    }
                    default: {
                        if (writer != null) break;
                        throw new IllegalArgumentException("Unrecognised image");
                    }
                }
                assert (writer != null);
                MemoryCacheImageOutputStream cacheImageOutputStream = new MemoryCacheImageOutputStream(out);
                writer.setOutput(cacheImageOutputStream);
                IIOImage outputImage = new IIOImage(transformedImage, null, null);
                writer.write(null, outputImage, writeParam);
                cacheImageOutputStream.flush();
                long duration = System.currentTimeMillis() - start;
                LOGGER.debug("MS to Resize image: {}", (Object)duration);
            }
            catch (Throwable throwable) {
                var12_11 = throwable;
                throw throwable;
            }
        }
        catch (Exception e) {
            LOGGER.error("Exception when resizing image, returning original image", (Throwable)e);
            in.copyTo(out);
        }
    }

    private static ImageReader getImageReader(ImageResizer.FileType fileType) {
        ImageReader reader;
        switch (fileType) {
            case gif: {
                reader = ImageIO.getImageReadersByFormatName("GIF").next();
                break;
            }
            case jpg: {
                reader = ImageIO.getImageReadersByFormatName("JPEG").next();
                break;
            }
            case png: {
                reader = ImageIO.getImageReadersByFormatName("PNG").next();
                break;
            }
            case webp: {
                reader = ImageIO.getImageReadersByFormatName("WEBP").next();
                break;
            }
            default: {
                throw new IllegalArgumentException("Unrecognised image");
            }
        }
        return reader;
    }

    public static AffineTransformOp getInverseTransformOp(int width, int height, ImageResizer.Orientation orientation) {
        AffineTransform tx = new AffineTransform();
        switch (orientation.rotationDegrees) {
            case 0: {
                break;
            }
            case 90: {
                tx.rotate(1.5707963267948966);
                tx.translate(0.0, -height);
                break;
            }
            case 180: {
                tx.rotate(Math.PI);
                tx.translate(-width, -height);
                break;
            }
            case 270: {
                tx.rotate(-1.5707963267948966);
                tx.translate(-width, 0.0);
                break;
            }
            default: {
                throw new IllegalArgumentException("Don't know how to handle rotationDegrees=" + orientation.rotationDegrees);
            }
        }
        if (orientation.mirrored) {
            tx.scale(-1.0, 1.0);
            tx.translate(-width, 0.0);
        }
        return new AffineTransformOp(tx, 2);
    }

    private BufferedImage transformImage(BufferedImage source, int maxWidthIn, int maxHeightIn, float width, float height, ImageResizer.Orientation orientation, RenderingHints hints) {
        int maxWidth = maxWidthIn;
        int maxHeight = maxHeightIn;
        if (orientation.rotationDegrees == 90 || orientation.rotationDegrees == 270) {
            maxWidth = maxHeightIn;
            maxHeight = maxWidthIn;
        }
        boolean tooWide = this.shouldResizeWidth(width, maxWidth);
        boolean tooHigh = this.shouldResizeHeight(height, maxHeight);
        if (tooWide || tooHigh || orientation != ImageResizer.Orientation.Normal) {
            float heightScale;
            float scale = 1.0f;
            if (tooWide) {
                scale = (float)maxWidth / width;
            }
            if (tooHigh && (heightScale = (float)maxHeight / height) < scale) {
                scale = heightScale;
            }
            int targetWidth = (int)(width * scale);
            int targetHeight = (int)(height * scale);
            ResampleOp scaleOp = new ResampleOp(targetWidth, targetHeight, hints);
            AffineTransformOp orientationOp = ImageIoResizer.getInverseTransformOp(targetWidth, targetHeight, orientation);
            return Scalr.apply((BufferedImage)source, (BufferedImageOp[])new BufferedImageOp[]{scaleOp, orientationOp});
        }
        return source;
    }

    @Override
    public long getResizedImageLength(ByteSource source, HashString hash, ZonedDateTime entityLastModified, int maxWidth, int maxHeight, ImageResizer.FileType fileType) throws IOException {
        LengthCountingOutputStream stream = new LengthCountingOutputStream();
        this.renderResized(source, hash, entityLastModified, stream, maxWidth, maxHeight, fileType);
        return stream.length();
    }

    private boolean shouldResizeWidth(float width, float maxWidth) {
        return width > maxWidth && maxWidth > 0.0f;
    }

    private boolean shouldResizeHeight(float height, float maxHeight) {
        return height > maxHeight && maxHeight > 0.0f;
    }

    public static Pair<Integer, Integer> getDimensions(InputStream input) throws IOException {
        ImageInputStream imageStream = null;
        ImageReader reader = null;
        try {
            imageStream = ImageIO.createImageInputStream(input);
            reader = ImageIO.getImageReaders(imageStream).next();
            reader.setInput(imageStream, true, true);
            Pair<Integer, Integer> pair = Pair.of(reader.getWidth(0), reader.getHeight(0));
            return pair;
        }
        catch (NoSuchElementException e) {
            throw new NotAnImageException(e);
        }
        finally {
            if (reader != null) {
                reader.dispose();
            }
            if (imageStream != null) {
                imageStream.close();
            }
            input.close();
        }
    }

    public boolean isOversized(int width, int height) {
        return width * height > this.maxWidthToResize * this.maxHeightToResize;
    }

    public static ImageResizer.Orientation getOrientation(InputStream input, ImageResizer.FileType fileType) {
        if (fileType == ImageResizer.FileType.webp) {
            return ImageIoResizer.getOrientationForWebpFile(input);
        }
        try {
            TiffField orientationField;
            TiffImageMetadata exif;
            ImageMetadata metadata = Imaging.getMetadata((InputStream)input, (String)("image." + fileType.name()));
            if (metadata instanceof JpegImageMetadata && (exif = ((JpegImageMetadata)metadata).getExif()) != null && (orientationField = exif.findField((TagInfo)TiffTagConstants.TIFF_TAG_ORIENTATION)) != null) {
                return ImageResizer.Orientation.fromExifOrientationValue(orientationField.getIntValue());
            }
        }
        catch (IOException | ImageReadException e) {
            LOGGER.warn("Couldn't read the image orientation", e);
        }
        return ImageResizer.Orientation.Normal;
    }

    private static ImageResizer.Orientation getOrientationForWebpFile(InputStream inputStream) {
        try {
            Metadata alternativeMetadata = ImageMetadataReader.readMetadata((InputStream)inputStream);
            ExifIFD0Directory directory = (ExifIFD0Directory)alternativeMetadata.getFirstDirectoryOfType(ExifIFD0Directory.class);
            if (directory != null) {
                return ImageResizer.Orientation.fromExifOrientationValue(directory.getInt(274));
            }
        }
        catch (ImageProcessingException | MetadataException | IOException e) {
            LOGGER.warn("Couldn't read the image orientation", e);
        }
        return ImageResizer.Orientation.Normal;
    }

    public void setMaxWidthToResize(int maxWidth) {
        this.maxWidthToResize = maxWidth;
    }

    public void setMaxHeightToResize(int maxHeight) {
        this.maxHeightToResize = maxHeight;
    }

    static class NotAnImageException
    extends IOException {
        private static final long serialVersionUID = -3845937464714363305L;

        NotAnImageException(Throwable t) {
            super(t);
        }
    }
}

