package uk.ac.warwick.util.mywarwick.model;

import java.time.Duration;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

abstract class AbstractSpringConfiguration implements Configuration {
  protected final Properties defaults;
  
  private Set<Instance> instanceSet;

  public AbstractSpringConfiguration() {
    this.defaults = Configuration.loadDefaults();
  }
  
  protected String getOrDefault(String key) {
    String value = getOrNull(key);
    if (value == null) return defaults.getProperty(key);
    return value;
  }
  
  abstract protected String getOrNull(String key);
  
  abstract boolean keyExists(String key);

  private void initMyWarwickConfigs() {
    
    Optional<Integer> firstMissingIndex = Stream.iterate(0, i -> i + 1)
      .filter(i -> !keyExists("mywarwick.instances."+i+".providerId"))
      .findFirst();
    
    if (!firstMissingIndex.isPresent()) {
      throw new IllegalArgumentException("No instances found under mywarwick.instances.*");
    }
    
    int instances = firstMissingIndex.get();
    
    instanceSet = Stream.iterate(0, i -> i + 1).limit(instances).map(i -> {
      String prefix = "mywarwick.instances." + i;
      return new Instance(
        getOrNull(prefix + ".baseUrl"),
        getOrNull(prefix + ".providerId"),
        getOrNull(prefix + ".userName"),
        getOrNull(prefix + ".password"),
        getOrNull(prefix + ".logErrors")
      );
    }).collect(Collectors.toSet());
  }

  public Set<Instance> getInstances() {
    if (instanceSet == null) {
      instanceSet = new HashSet<>();
      initMyWarwickConfigs();
    }
    return instanceSet;
  }

  public int getHttpMaxConn() {
    return Integer.parseInt(getOrDefault("mywarwick.http.maxConn"));
  }

  public int getHttpMaxConnPerRoute() {
    return Integer.parseInt(getOrDefault("mywarwick.http.maxConnPerRoute"));
  }

  public String getQuartzHealthcheckName() {
    return getOrDefault("mywarwick.quartz.healthcheck.name");
  }

  public long getQuartzHealthcheckQueueSizeWarningThreshold() {
    return Long.parseLong(getOrDefault("mywarwick.quartz.healthcheck.queueSize.warning"));
  }

  public long getQuartzHealthcheckQueueSizeCriticalThreshold() {
    return Long.parseLong(getOrDefault("mywarwick.quartz.healthcheck.queueSize.critical"));
  }

  public Duration getQuartzHealthcheckDelayWarningThreshold() {
    return Duration.parse(getOrDefault("mywarwick.quartz.healthcheck.delay.warning"));
  }

  public Duration getQuartzHealthcheckDelayCriticalThreshold() {
    return Duration.parse(getOrDefault("mywarwick.quartz.healthcheck.delay.critical"));
  }
}
