package uk.ac.warwick.util.mywarwick;

import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.warwick.util.core.DateTimeUtils;
import uk.ac.warwick.util.mywarwick.model.Configuration;
import uk.ac.warwick.util.mywarwick.model.Instance;
import uk.ac.warwick.util.mywarwick.model.response.Response;

import javax.inject.Inject;
import java.sql.Date;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;

/**
 * Send a single My Warwick activity, this may do retries.
 */
public class SendMyWarwickActivityJob implements Job {

    public static final JobKey JOB_KEY = new JobKey("SendActivityJob", "warwickutils-mywarwick");

    public static final String INSTANCE_BASE_URL_DATA_KEY = "instanceBaseUrl";

    public static final String IS_NOTIFICATION_JOB_DATA_KEY = "isNotification";

    public static final String IS_TRANSIENT_JOB_DATA_KEY = "isTransient";

    public static final String REQUEST_BODY_JOB_DATA_KEY = "requestBody";

    public static final String CREATED_DATETIME_ISO8601_DATA_KEY = "createdDateTimeISO8601";

    private static final Logger LOGGER = LoggerFactory.getLogger(SendMyWarwickActivityJob.class);

    private static final Duration RESCHEDULE_DELAY = Duration.ofSeconds(30);

    private final MyWarwickService myWarwickService;

    private final Scheduler scheduler;

    private final Set<Instance> instances;

    @Inject
    public SendMyWarwickActivityJob(MyWarwickService myWarwickService, Scheduler scheduler, Configuration configuration) {
        this.myWarwickService = myWarwickService;
        this.scheduler = scheduler;
        this.instances = configuration.getInstances();
    }

    private void reschedule(JobExecutionContext context) throws JobExecutionException {
        try {
            scheduler.rescheduleJob(
                context.getTrigger().getKey(),
                TriggerBuilder.newTrigger()
                    .withIdentity(context.getTrigger().getKey())
                    .startAt(Date.from(Instant.now(DateTimeUtils.CLOCK_IMPLEMENTATION).plusSeconds(RESCHEDULE_DELAY.getSeconds())))
                    .usingJobData(context.getTrigger().getJobDataMap())
                    .build()
            );
        } catch (SchedulerException e) {
            throw new JobExecutionException(e);
        }
    }

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        JobDataMap jobDataMap = context.getMergedJobDataMap();

        String instanceBaseUrl = jobDataMap.getString(INSTANCE_BASE_URL_DATA_KEY);
        if (instanceBaseUrl == null) {
            // Missing data, schedule a new job-per-instance
            for (Instance instance : instances) {
                try {
                    JobDataMap data = new JobDataMap(context.getTrigger().getJobDataMap());
                    data.put(SendMyWarwickActivityJob.INSTANCE_BASE_URL_DATA_KEY, instance.getBaseUrl());

                    scheduler.scheduleJob(
                        TriggerBuilder.newTrigger()
                            .forJob(SendMyWarwickActivityJob.JOB_KEY)
                            .usingJobData(data)
                            .build()
                    );
                } catch (SchedulerException e) {
                    throw new JobExecutionException(e);
                }
            }
            return;
        }

        Optional<Instance> requestedInstance =
            instances.stream()
                .filter(i -> i.getBaseUrl().equals(instanceBaseUrl))
                .findFirst();

        if (requestedInstance.isPresent()) {
            Instance instance = requestedInstance.get();
            String requestBody = jobDataMap.getString(REQUEST_BODY_JOB_DATA_KEY);
            boolean isNotification = jobDataMap.getBooleanValueFromString(IS_NOTIFICATION_JOB_DATA_KEY);
            boolean isTransient = jobDataMap.getBooleanValueFromString(IS_TRANSIENT_JOB_DATA_KEY);

            try {
                Response response =
                    myWarwickService.sendSingleInstance(instance, requestBody, isNotification, isTransient, MyWarwickService.DEFAULT_MAX_ATTEMPTS)
                        .join();

                if (Boolean.TRUE.equals(response.getSuccess())) {
                    LOGGER.trace("Sent My Warwick activity " + requestBody);
                } else if (Boolean.FALSE.equals(response.getSuccess())) {
                    String errorMessage =
                            response.getErrors().stream().map(e -> {
                                if (e.getMessage() != null && !e.getMessage().isEmpty()) {
                                    return e.getMessage();
                                }
                                return e.getId();
                            }).collect(Collectors.joining(", "));

                    if (response.getErrors().stream().anyMatch(e -> e.getId().equals("no-permission"))) {
                        doLog(instance, String.format(
                                "Missing permission to send My Warwick activity (%S): %s - retrying in %ss",
                                response.getStatus(),
                                errorMessage,
                                RESCHEDULE_DELAY.getSeconds()
                        ));
                        reschedule(context);
                    } else {
                        doLog(instance, String.format(
                                "Validation error sending My Warwick activity (%s): %s - cancelling",
                                response.getStatus(),
                                errorMessage
                        ));
                    }
                } else {
                    doLog(instance, String.format(
                            "Unexpected response from My Warwick (%s) - retrying in %ss",
                            response.getStatus(),
                            RESCHEDULE_DELAY.getSeconds()
                    ));
                    reschedule(context);
                }
            } catch (CancellationException e) {
                doLog(instance, String.format(
                        "CompletableFuture cancelled while sending My Warwick activity, retrying in %ss",
                        RESCHEDULE_DELAY.getSeconds()
                ), e);
                reschedule(context);
            } catch (CompletionException e) {
                doLog(instance, String.format(
                        "Exception sending activity, retrying in %ss",
                        RESCHEDULE_DELAY.getSeconds()
                ), e);
                reschedule(context);
            }
        } else {
            LOGGER.error("Error sending My Warwick activity, instance " + instanceBaseUrl + " couldn't be found, cancelling");
        }
    }

    private void doLog(Instance instance, String message) {
        if (instance.isLogErrors()) {
            LOGGER.error(String.format("%s: %s", instance.getBaseUrl(), message));
        } else {
            LOGGER.warn(String.format("%s: %s", instance.getBaseUrl(), message));
        }
    }

    private void doLog(Instance instance, String message, Throwable e) {
        if (instance.isLogErrors()) {
            LOGGER.error(String.format("%s: %s", instance.getBaseUrl(), message), e);
        } else {
            LOGGER.warn(String.format("%s: %s", instance.getBaseUrl(), message), e);
        }
    }
}
