package uk.ac.warwick.util.mywarwick;

import org.quartz.Scheduler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.util.ClassUtils;
import uk.ac.warwick.util.mywarwick.healthcheck.SpringMyWarwickQuartzHealthcheckProvider;

import javax.inject.Named;
import java.io.IOException;
import java.util.*;

/**
 * this is a configuration bean for Spring (version >= 3) application
 * use it like this in context XML:
 * <pre>
 * {@code
 * <bean class="uk.ac.warwick.util.mywarwick.MyWarwickSpringConfig" />
 * }
 * </pre>
 *
 * To provide properties, you need to either:
 * <ul>
 *   <li>Provide a properties bean with id "applicationPropertiesForMyWarwick".
 *       If you have an existing Properties bean, you can use the alias element to alias it to the above ID.</li>
 *   <li>Define a {@link uk.ac.warwick.util.mywarwick.model.Configuration} bean yourself</li>
 * </ul>
 */
@Configuration
@ComponentScan("uk.ac.warwick.util.mywarwick")
public class MyWarwickSpringConfig {

    /**
     * this bean is depended by {@link uk.ac.warwick.util.mywarwick.MyWarwickServiceImpl}
     * @return {@link uk.ac.warwick.util.mywarwick.model.Configuration} which is implemented by {@link uk.ac.warwick.util.mywarwick.model.PropertiesConfiguration}
     * @throws IOException when properties cannot be loaded from context
     */
    @Bean
    @Conditional(ConfigurationMissingCondition.class)
    public uk.ac.warwick.util.mywarwick.model.Configuration myWarwickServiceConfiguration(
      @Named(value = "applicationPropertiesForMyWarwick")
      Properties propertiesBean
    ) {
        return new uk.ac.warwick.util.mywarwick.model.PropertiesConfiguration(propertiesBean);
    }

    @Configuration
    @Conditional(HealthcheckClassesPresent.class)
    @Profile("scheduling")
    public static class HealthcheckConfig {
        @Bean
        public SpringMyWarwickQuartzHealthcheckProvider myWarwickQuartzHealthcheckProvider(@Autowired Scheduler scheduler, @Autowired uk.ac.warwick.util.mywarwick.model.Configuration myWarwickServiceConfiguration) {
            return new SpringMyWarwickQuartzHealthcheckProvider(scheduler, myWarwickServiceConfiguration);
        }
    }

    /**
     * Condition to pass only if a Configuration isn't already defined as a bean. Allows an application to provide
     * their own implementation.
     * 
     * Would use @ConditionalOnMissingBean but that's a Spring Boot thing.
     */
    public static class ConfigurationMissingCondition implements ConfigurationCondition {
        
        private static final List<String> configurationClassNames = new ArrayList<String>() {{
            add("uk.ac.warwick.util.mywarwick.model.TypesafeConfiguration");
            add("uk.ac.warwick.util.mywarwick.model.SpringEnvironmentConfiguration");
            add("uk.ac.warwick.util.mywarwick.model.PropertiesConfiguration");
        }};

        @Override
        public boolean matches(ConditionContext conditionContext, AnnotatedTypeMetadata annotatedTypeMetadata) {
            String[] beanNames = conditionContext.getRegistry().getBeanDefinitionNames();
            return Arrays.stream(beanNames).noneMatch(name -> {
                BeanDefinition def = conditionContext.getRegistry().getBeanDefinition(name);
                return configurationClassNames.stream().anyMatch(configClassName -> Objects.equals(configClassName, def.getBeanClassName()));
            });
        }

        @Override
        public ConfigurationPhase getConfigurationPhase() {
            return ConfigurationPhase.REGISTER_BEAN;
        }
    }

    public static class HealthcheckClassesPresent implements Condition {
        private static final Logger LOGGER = LoggerFactory.getLogger(HealthcheckClassesPresent.class);

        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            ClassLoader classLoader = context.getClassLoader();
            if (classLoader == null) {
                classLoader = ClassUtils.getDefaultClassLoader();
            }

            try {
                Class.forName("uk.ac.warwick.util.service.ServiceHealthcheckProvider", true, classLoader);
                Class.forName("org.quartz.Scheduler", true, classLoader);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("HealthcheckClassesPresent condition resolving to true");
                }
                return true;
            } catch (Throwable t) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("HealthcheckClassesPresent condition resolving to false", t);
                }
                return false;
            }
        }
    }
}
