package uk.ac.warwick.util.mywarwick;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import uk.ac.warwick.util.mywarwick.model.Instance;

import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.util.concurrent.CompletableFuture;

public class HasRegistrationHttpResponseCallback implements FutureCallback<HttpResponse> {

    private final String reqPath;
    private final Instance myWarwickInstance;
    private final Logger logger;
    private final CompletableFuture<Boolean> completableFuture;

    HasRegistrationHttpResponseCallback(
            @NotNull String reqPath,
            @NotNull Instance myWarwickInstance,
            @NotNull Logger logger,
            @NotNull CompletableFuture<Boolean> completableFuture
    ) {
        super();
        this.reqPath = reqPath;
        this.myWarwickInstance = myWarwickInstance;
        this.logger = logger;
        this.completableFuture = completableFuture;
    }

    @Override
    public void completed(HttpResponse httpResponse) {
        if (logger.isDebugEnabled()) logger.debug("Request completed");
        try {
            final ObjectNode node = new ObjectMapper().readValue(EntityUtils.toString(httpResponse.getEntity()), ObjectNode.class);
            if (node.has("data")) {
                completeFuture(node.get("data").asBoolean());
                return;
            }
            completeFuture(false);
        } catch (IOException e) {
            logError("An IOException was thrown communicating with mywarwick:\n" +
                    e.getMessage() +
                    "\nbaseUrl: " + myWarwickInstance.getBaseUrl());
            completeFuture(false);
        }
    }

    @Override
    public void failed(Exception e) {
        completeFuture(false);
    }

    @Override
    public void cancelled() {
        completeFuture(false);
    }

    private void logError(String message) {
        if (myWarwickInstance.isLogErrors()) {
            logger.error(message);
        } else {
            logger.warn(message);
        }
    }

    private void logError(String message, Exception e) {
        if (myWarwickInstance.isLogErrors()) {
            logger.error(message, e);
        } else {
            logger.warn(message, e);
        }
    }

    private void completeFuture(boolean response) {
        completableFuture.complete(response);
    }

}
