package uk.ac.warwick.util.files.impl;

import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.BlobServiceClientBuilder;
import org.jclouds.Constants;
import org.jclouds.ContextBuilder;
import org.jclouds.azureblob.AzureBlobApiMetadata;
import org.jclouds.blobstore.BlobStore;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.TransientApiMetadata;
import org.jclouds.filesystem.FilesystemApiMetadata;
import org.jclouds.filesystem.reference.FilesystemConstants;
import org.jclouds.logging.slf4j.config.SLF4JLoggingModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.AbstractFactoryBean;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import uk.ac.warwick.util.core.StringUtils;

import java.io.File;
import java.util.Collections;
import java.util.Properties;

import static java.lang.String.format;

public class BlobStoreContextFactoryBean extends AbstractFactoryBean<BlobStoreContext> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BlobStoreContextFactoryBean.class);

    @Value("${objectstore.provider:swift}")
    private String providerType;

    @Value("${objectstore.swift.endpoint:}")
    private String swiftEndpoint;

    @Value("${objectstore.swift.username:}")
    private String swiftUsername;

    @Value("${objectstore.swift.password:}")
    private String swiftPassword;

    @Value("${objectstore.azure.endpoint:}")
    private String azureEndpoint;

    @Value("${objectstore.azure.username:}")
    private String azureUsername;

    @Value("${objectstore.azure.password:}")
    private String azurePassword;

    @Value("${objectstore.fileSystem.root:}")
    private String fileSystemRoot;

    private Properties overrides = new Properties();

    @Override
    public Class<BlobStoreContext> getObjectType() {
        return BlobStoreContext.class;
    }

    @Override
    protected BlobStoreContext createInstance() throws Exception {
        switch (providerType) {
            case "inMemory":
                return
                    ContextBuilder.newBuilder(new TransientApiMetadata())
                        .modules(Collections.singleton(new SLF4JLoggingModule()))
                        .buildView(BlobStoreContext.class);
            case "fileSystem":
                return getFileSystemContext();
            case "azure":
                return getAzureContext();
            case "swift":
                return getSwiftContext();
            case "swift+azure":
                return getSwiftPlusAzureContext();
            default:
                LOGGER.warn("Unknown objectstore.provider '" + providerType + "'; defaulting to swift");
                return getSwiftContext();
        }
    }

    /**
     * jClouds context for access to Azure proper.
     */
    private BlobStoreContext getAzureContext() {
        // endpoint is optional if using a production account as the endpoint is guessable from the username/account
        //Assert.isTrue(StringUtils.hasText(azureEndpoint), "The property objectstore.azure.endpoint must be non-empty");
        Assert.isTrue(StringUtils.hasText(azureUsername), "The property objectstore.azure.username must be non-empty");
        Assert.isTrue(StringUtils.hasText(azurePassword), "The property objectstore.azure.password must be non-empty");

        String endpoint = getAzureEndpoint();

        if (!ClassUtils.isPresent("org.jclouds.azureblob.AzureBlobApiMetadata", null)) {
            LOGGER.warn("AzureBlobApiMetadata not found - if you see an error after this, you probably need to add the jclouds azureblob provider to your classpath.");
        }

        final BlobStoreContext realContext = ContextBuilder.newBuilder("azureblob")
          .endpoint(endpoint)
          .credentials(azureUsername, azurePassword)
          .modules(Collections.singleton(new SLF4JLoggingModule()))
          .overrides(overrides)
          .buildView(BlobStoreContext.class);

        return new DelegatingBlobStoreContext(realContext) {
            public BlobStore getBlobStore() {
                // Need to wrap the context to return a wrapped blobstore that can translate container names,
                // since we use dots extensively which Azure doesn't allow.
                return new ContainerRenamingBlobStore(this, realContext.getBlobStore(), RenameMode.NO_DOTS);
            }
        };
    }

    private String getAzureEndpoint() {
        return StringUtils.hasText(azureEndpoint) ? azureEndpoint : format("https://%s.blob.core.windows.net/", azureUsername);
    }

    private BlobStoreContext getSwiftContext() {
        Assert.isTrue(StringUtils.hasText(swiftEndpoint), "The property objectstore.swift.endpoint must be non-empty");
        Assert.isTrue(StringUtils.hasText(swiftUsername), "The property objectstore.swift.username must be non-empty");
        Assert.isTrue(StringUtils.hasText(swiftPassword), "The property objectstore.swift.password must be non-empty");

        return
            ContextBuilder.newBuilder("openstack-swift")
                .endpoint(swiftEndpoint)
                .credentials(format("LDAP_%s:%s", swiftUsername, swiftUsername), swiftPassword)
                .modules(Collections.singleton(new SLF4JLoggingModule()))
                .buildView(BlobStoreContext.class);
    }

    private BlobStoreContext getSwiftPlusAzureContext() {
        Assert.isTrue(StringUtils.hasText(azureUsername), "The property objectstore.azure.username must be non-empty");
        Assert.isTrue(StringUtils.hasText(azurePassword), "The property objectstore.azure.password must be non-empty");

        if (!ClassUtils.isPresent("com.azure.storage.blob.BlobServiceClient", null)) {
            throw new IllegalStateException("Swift+Azure requires com.azure:azure-storage-blob to be on the classpath");
        }

        return new WriteToAzureBlobStoreContext(getSwiftContext(), AzureClientFactory.getAzureClient(azureUsername, azurePassword, getAzureEndpoint()));
    }

    private BlobStoreContext getFileSystemContext() {
        Assert.isTrue(StringUtils.hasText(fileSystemRoot), "The property objectstore.fileSystem.root must be non-empty");
        File root = new File(fileSystemRoot);
        Assert.isTrue((root.exists() || root.mkdirs()) && root.isDirectory(), "The root directory " + root + " must exist and be a directory");

        return
          ContextBuilder.newBuilder(new FilesystemApiMetadata())
            .overrides(new Properties() {{
                setProperty(FilesystemConstants.PROPERTY_BASEDIR, root.getAbsolutePath());
            }})
            .modules(Collections.singleton(new SLF4JLoggingModule()))
            .buildView(BlobStoreContext.class);
    }

    /**
     * Set overrides which are passed directly to the jclouds context builder.
     */
    public void setOverrides(Properties overrides) {
        this.overrides = overrides;
    }

    @Override
    protected void destroyInstance(BlobStoreContext context) throws Exception {
        if (context != null) context.close();
    }

}

/**
 * Class boundary so that Azure SDK doesn't have to be on the classpath unless you're using it.
 */
class AzureClientFactory {
    /**
     * Azure SDK client for use in swift+azure migration mode, since the jclouds API
     * doesn't support the required operations such as creating blobs from URLs.
     */
    static BlobServiceClient getAzureClient(String accountName, String accountKey, String endpoint) {
        String connectionString = format(
          "DefaultEndpointsProtocol=https;AccountName=%s;AccountKey=%s;BlobEndpoint=%s",
          accountName,
          accountKey,
          endpoint
        );

        return new BlobServiceClientBuilder()
          .connectionString(connectionString)
          .buildClient();
    }
}
