/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.warwick.util.termdates;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Ordering;
import java.io.Serializable;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.Month;
import java.time.YearMonth;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.threeten.extra.LocalDateRange;
import uk.ac.warwick.util.termdates.AcademicWeek;
import uk.ac.warwick.util.termdates.AcademicYearPeriod;
import uk.ac.warwick.util.termdates.TermDatesService;

public class AcademicYear
implements Comparable<AcademicYear>,
Serializable {
    static final Pattern SITS_PATTERN = Pattern.compile("(\\d{2})/(\\d{2})");
    static final int CENTURY_BREAK = 90;
    private final int startYear;
    private final Map<AcademicYearPeriod.PeriodType, AcademicYearPeriod> periods;
    private final Map<Integer, AcademicWeek> weeks;

    protected AcademicYear(int startYear, List<AcademicYearPeriod> periods) {
        AcademicYear.verify(startYear >= 1000 && startYear < 9999, "Invalid start year: " + startYear);
        this.startYear = startYear;
        this.periods = periods.stream().map(period -> period.withYear(this)).collect(Collectors.toMap(AcademicYearPeriod::getType, Function.identity()));
        this.weeks = AcademicYear.buildWeeks(this, this.periods);
    }

    private static Map<Integer, AcademicWeek> buildWeeks(AcademicYear year, Map<AcademicYearPeriod.PeriodType, AcademicYearPeriod> periods) {
        if (periods.isEmpty()) {
            return Collections.emptyMap();
        }
        ImmutableMap.Builder weeks = ImmutableMap.builder();
        LocalDate firstDayOfAutumnTerm = periods.get((Object)AcademicYearPeriod.PeriodType.autumnTerm).getFirstDay();
        LocalDate firstDayOfAcademicYear = firstDayOfAutumnTerm.with(Month.AUGUST).withDayOfMonth(1);
        int offset = (int)ChronoUnit.WEEKS.between(firstDayOfAcademicYear, firstDayOfAutumnTerm);
        if (firstDayOfAcademicYear.getDayOfWeek().equals(DayOfWeek.MONDAY)) {
            --offset;
        }
        int weekNumber = -offset;
        for (AcademicYearPeriod period : AcademicYear.getSortedValues(periods)) {
            LocalDate start = period.getFirstDay();
            LocalDate end = start.plusWeeks(1L);
            if (!start.getDayOfWeek().equals(DayOfWeek.MONDAY)) {
                end = end.with(DayOfWeek.MONDAY);
            }
            while (end.isBefore(period.getLastDay())) {
                weeks.put((Object)weekNumber, (Object)AcademicWeek.of(year, period, weekNumber, LocalDateRange.of((LocalDate)start, (LocalDate)end)));
                start = end;
                end = start.plusWeeks(1L);
                ++weekNumber;
            }
            end = period.getLastDay().plusDays(1L);
            weeks.put((Object)weekNumber, (Object)AcademicWeek.of(year, period, weekNumber, LocalDateRange.of((LocalDate)start, (LocalDate)end)));
            ++weekNumber;
        }
        return weeks.build();
    }

    static AcademicYear build(AcademicYearPeriod ... periods) {
        AcademicYear.verify(periods.length > 0, "Must provide at least one AcademicYearPeriod");
        List<AcademicYearPeriod> sortedPeriods = Collections.unmodifiableList(Ordering.natural().sortedCopy(Arrays.asList(periods)));
        AcademicYear.verify(sortedPeriods.size() == AcademicYearPeriod.PeriodType.values().length, "Must provide an AcademicYearPeriod of every type");
        for (AcademicYearPeriod.PeriodType periodType : AcademicYearPeriod.PeriodType.values()) {
            AcademicYear.verify(sortedPeriods.stream().anyMatch(p -> p.getType() == periodType));
        }
        return new AcademicYear(((AcademicYearPeriod)sortedPeriods.iterator().next()).getFirstDay().getYear(), sortedPeriods);
    }

    static AcademicYear placeholder(int startYear) {
        return new AcademicYear(startYear, Collections.emptyList());
    }

    public boolean isPlaceholder() {
        return this.periods.isEmpty();
    }

    public static AcademicYear starting(int startYear) {
        return TermDatesService.INSTANCE.getAcademicYear(startYear);
    }

    public static AcademicYear forDate(Temporal temporal) {
        YearMonth yearMonth = YearMonth.from(temporal);
        return yearMonth.getMonthValue() < Month.AUGUST.getValue() ? AcademicYear.starting(yearMonth.getYear() - 1) : AcademicYear.starting(yearMonth.getYear());
    }

    public static AcademicYear parse(String pattern) {
        Matcher m = SITS_PATTERN.matcher(pattern);
        if (m.matches()) {
            int startYear = Integer.parseInt(m.group(1));
            return startYear > 90 ? AcademicYear.starting(1900 + startYear) : AcademicYear.starting(2000 + startYear);
        }
        throw new IllegalArgumentException("Did not match YY/YY: " + pattern);
    }

    public int getStartYear() {
        return this.startYear;
    }

    private static <K, V extends Comparable<? super V>> List<V> getSortedValues(Map<K, V> map) {
        return map.entrySet().stream().sorted(Map.Entry.comparingByValue()).map(Map.Entry::getValue).collect(Collectors.toList());
    }

    public List<AcademicYearPeriod> getPeriods() {
        if (this.periods.isEmpty()) {
            throw new IllegalStateException("No term dates found for " + this.toString());
        }
        return AcademicYear.getSortedValues(this.periods);
    }

    public AcademicYearPeriod getPeriod(AcademicYearPeriod.PeriodType type) {
        if (this.periods.isEmpty()) {
            throw new IllegalStateException("No term dates found for " + this.toString());
        }
        return this.periods.get((Object)type);
    }

    public AcademicYearPeriod getPeriod(Temporal date) {
        return this.getAcademicWeek(date).getPeriod();
    }

    public List<AcademicWeek> getAcademicWeeks() {
        if (this.periods.isEmpty()) {
            throw new IllegalStateException("No term dates found for " + this.toString());
        }
        return AcademicYear.getSortedValues(this.weeks);
    }

    public AcademicWeek getAcademicWeek(int weekNumber) {
        if (this.periods.isEmpty()) {
            throw new IllegalStateException("No term dates found for " + this.toString());
        }
        if (!this.weeks.containsKey(weekNumber)) {
            throw new IllegalArgumentException("Invalid week number: " + weekNumber + " for " + this.toString());
        }
        return this.weeks.get(weekNumber);
    }

    public AcademicWeek getAcademicWeek(Temporal date) {
        if (this.periods.isEmpty()) {
            throw new IllegalStateException("No term dates found for " + this.toString());
        }
        return this.getAcademicWeeks().stream().filter(w -> w.getDateRange().contains(LocalDate.from(date))).findFirst().orElseThrow(() -> new IllegalArgumentException("No week found for " + date));
    }

    public int getValue() {
        return this.getStartYear();
    }

    public String getLabel() {
        return this.toString();
    }

    public AcademicYear previous() {
        return AcademicYear.starting(this.startYear - 1);
    }

    public AcademicYear next() {
        return AcademicYear.starting(this.startYear + 1);
    }

    public List<AcademicYear> yearsSurrounding(int yearsBefore, int yearsAfter) {
        AcademicYear.verify(yearsBefore >= 0 && yearsAfter >= 0);
        ImmutableList.Builder years = ImmutableList.builder();
        for (int year = this.startYear - yearsBefore; year <= this.startYear + yearsAfter; ++year) {
            years.add((Object)AcademicYear.starting(year));
        }
        return years.build();
    }

    public boolean isBefore(AcademicYear other) {
        return this.compareTo(other) < 0;
    }

    public boolean isAfter(AcademicYear other) {
        return this.compareTo(other) > 0;
    }

    @Override
    public int compareTo(AcademicYear o) {
        return this.startYear - o.startYear;
    }

    public int hashCode() {
        return this.toString().hashCode();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        AcademicYear that = (AcademicYear)o;
        return this.startYear == that.startYear;
    }

    public String toString() {
        return String.format("%s/%s", Integer.toString(this.startYear).substring(2), Integer.toString(this.startYear + 1).substring(2));
    }

    protected static void verify(boolean condition) {
        if (!condition) {
            throw new IllegalArgumentException();
        }
    }

    protected static void verify(boolean condition, String message) {
        if (!condition) {
            throw new IllegalArgumentException(message);
        }
    }
}

