/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.warwick.util.core.spring;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.DecimalFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.FileCopyUtils;
import uk.ac.warwick.util.core.StringUtils;

public final class FileUtils {
    public static final String BYTE_LABEL = "B";
    public static final String KILO_BYTE_LABEL = "KB";
    public static final String MEGA_BYTE_LABEL = "MB";
    public static final String GIGA_BYTE_LABEL = "GB";
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtils.class);
    private static final int TEMP_FILE_RETRIES = 5;
    private static final float MAX_MB_TO_SHOW_DECIMALS = 10.0f;
    private static final float MAX_GB_TO_SHOW_DECIMALS = 10.0f;
    private static final int KB_IN_MB = 1024;
    private static final int BYTES_IN_KB = 1024;
    private static final int MB_IN_GB = 1024;

    private FileUtils() {
        Assert.isTrue((boolean)true);
    }

    public static void recursiveDelete(File file) {
        FileUtils.recursiveDelete(file, true, null);
    }

    public static void recursiveDelete(File file, boolean strict) {
        FileUtils.recursiveDelete(file, strict, null);
    }

    public static void recursiveDelete(File file, File deletionBin) {
        FileUtils.recursiveDelete(file, true, deletionBin);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static void recursiveDelete(File file, boolean strict, File deletionBin) {
        boolean bl;
        if (file.isDirectory()) {
            if (LOGGER.isDebugEnabled()) {
                String string2 = "Directory contains files (pre-delete):";
                string2 = string2 + FileUtils.recursiveOutput(file);
                LOGGER.debug(string2);
            }
            for (File child : file.listFiles()) {
                FileUtils.recursiveDelete(child, strict, deletionBin);
            }
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Deleting " + file);
        }
        if (bl = file.delete()) return;
        if (strict) {
            LOGGER.info("Could not delete the " + (file.isDirectory() ? "directory" : "file") + " " + file);
            if (LOGGER.isDebugEnabled() && file.isDirectory()) {
                if (file.list().length == 0) {
                    LOGGER.debug("Directory contains no files anymore though");
                } else {
                    String logMessage = "Directory contains files:";
                    logMessage = logMessage + FileUtils.recursiveOutput(file);
                    LOGGER.debug(logMessage);
                }
            }
            if (deletionBin == null || !file.isDirectory()) throw new IllegalStateException("Cannot delete " + file);
            if (!deletionBin.isDirectory() || !deletionBin.exists()) {
                throw new IllegalArgumentException("Deletion bin " + deletionBin + " must be an existing directory");
            }
            File renameToFile = new File(deletionBin, file.getName() + System.currentTimeMillis());
            if (renameToFile.exists()) {
                throw new IllegalStateException("Could not rename directory to " + renameToFile + " - file already exists");
            }
            boolean success = file.renameTo(renameToFile);
            if (!success) {
                throw new IllegalStateException("Failed to rename directory to " + renameToFile);
            }
            renameToFile.deleteOnExit();
            return;
        } else {
            file.deleteOnExit();
            LOGGER.info("Could not delete the file " + file + ", marked to delete on exit");
        }
    }

    public static void copy(File source, File target, boolean recurse) throws IOException {
        FileUtils.sanityCheck(source, target, recurse);
        if (source.isDirectory()) {
            File[] filesInDir = source.listFiles();
            if (!target.mkdirs() && !target.exists()) {
                throw new IOException("Cannot create " + target);
            }
            for (File child : filesInDir) {
                if (!child.isFile() && !recurse) continue;
                FileUtils.copy(child, new File(target, child.getName()), recurse);
            }
        } else {
            if (!target.getParentFile().exists() && !target.getParentFile().mkdirs()) {
                throw new IOException("Cannot create " + target);
            }
            FileCopyUtils.copy((File)source, (File)target);
        }
    }

    private static void sanityCheck(File source, File target, boolean recurse) {
        if (!source.exists()) {
            throw new IllegalArgumentException("The source " + source + " must exist.");
        }
        if (target.exists() && !target.isDirectory()) {
            throw new IllegalArgumentException("The target " + target + " must not exist.");
        }
        if (target.exists() && source.isDirectory() != target.isDirectory()) {
            throw new IllegalArgumentException(String.format("Source '%s' and target '%s' must both be directories or both be files", source, target));
        }
        if (recurse && target.getAbsolutePath().startsWith(source.getAbsolutePath() + "/")) {
            throw new IllegalArgumentException(String.format("Target '%s' cannot be a descendant of source '%s'", target, source));
        }
    }

    public static String getFileName(String url) {
        File file = new File(url);
        return file.getName();
    }

    public static String getFileNameWithoutExtension(String s) {
        int indexOfLastDot = StringUtils.safeSubstring(s, 1).lastIndexOf(46);
        if (indexOfLastDot < 0) {
            return s;
        }
        return s.substring(0, indexOfLastDot + 1);
    }

    private static String getExtension(String s) {
        int indexOfLastDot = StringUtils.safeSubstring(s, 1).lastIndexOf(46);
        if (indexOfLastDot < 0) {
            return "";
        }
        return s.substring(indexOfLastDot + 2);
    }

    public static String getLowerCaseExtension(String filename) {
        return FileUtils.getExtension(filename).toLowerCase();
    }

    public static boolean extensionMatches(String filename, String extension) {
        String compareExtension = extension.toLowerCase();
        compareExtension = compareExtension.replaceAll("[^\\.]*\\.", "");
        return FileUtils.getLowerCaseExtension(filename).equalsIgnoreCase(compareExtension);
    }

    public static String convertToSafeFileName(String originalName) {
        char c;
        if (!StringUtils.hasText(originalName)) {
            return originalName;
        }
        File file = new File(originalName.toLowerCase());
        String s = file.getName();
        String fileName = FileUtils.getFileNameWithoutExtension(s);
        String extension = FileUtils.getLowerCaseExtension(s);
        StringBuffer fileNameSB = new StringBuffer();
        for (byte b : StringUtils.create(fileName)) {
            c = (char)b;
            if (!FileUtils.isValidForFileName(c)) continue;
            fileNameSB.append(c);
        }
        while (fileNameSB.length() > 0 && fileNameSB.charAt(0) == '.') {
            fileNameSB.deleteCharAt(0);
        }
        if (StringUtils.hasText(extension)) {
            fileNameSB.append(".");
            for (byte b : StringUtils.create(extension)) {
                c = (char)b;
                if (!FileUtils.isValidForFileName(c)) continue;
                fileNameSB.append(c);
            }
        }
        String safeFileName = fileNameSB.toString();
        safeFileName = safeFileName.replaceAll(" ", "_");
        return safeFileName;
    }

    private static boolean isValidForFileName(char c) {
        return c >= 'a' && c <= 'z' || c >= 'A' && c <= 'Z' || c >= '0' && c <= '9' || c == '_' || c == '-' || c == ' ' || c == '.';
    }

    public static File createOrLoadFile(File parent, String fileName) {
        if (!parent.isDirectory()) {
            throw new IllegalStateException("Parent " + parent + " must be a directory");
        }
        File file = new File(parent, fileName);
        if (!file.exists()) {
            try {
                if (!file.createNewFile()) {
                    throw new IllegalStateException("Cannot create file " + file);
                }
            }
            catch (IOException e) {
                throw new IllegalStateException("Cannot create file " + file);
            }
        }
        return file;
    }

    public static String recursiveOutput(File file) {
        if (!file.exists()) {
            return "";
        }
        String output = "\n" + file.getPath();
        if (file.isDirectory()) {
            File[] subfiles;
            for (File subfile : subfiles = file.listFiles()) {
                output = output + FileUtils.recursiveOutput(subfile);
            }
        } else {
            output = output + " ";
            output = output + file.length();
        }
        return output;
    }

    public static File createFile(String theSuggestedName, InputStream theContents, File directory) throws IllegalStateException {
        File file = FileUtils.createFile(theSuggestedName, directory);
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(file);
            FileCopyUtils.copy((InputStream)theContents, (OutputStream)fos);
        }
        catch (IOException e) {
            throw new IllegalStateException("cannot copy contents [" + theContents + "] into " + file, e);
        }
        finally {
            if (fos != null) {
                try {
                    fos.close();
                }
                catch (IOException e) {
                    LOGGER.warn("Couldn't close OutputStream", (Throwable)e);
                }
            }
        }
        return file;
    }

    public static File createFile(String theSuggestedName, File directory) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new IllegalStateException("Unable to create directories for temporary file storage");
        }
        File file = null;
        for (int i = 0; i < 5; ++i) {
            file = FileUtils.generateRandomFile(theSuggestedName, directory);
            if (file.exists()) continue;
            try {
                file.createNewFile();
            }
            catch (IOException e) {
                throw new IllegalStateException("cannot create file for " + file);
            }
            return file;
        }
        throw new IllegalStateException("cannot create file for " + file);
    }

    private static File generateRandomFile(String filePrefix, File directory) {
        String fileName = filePrefix + System.nanoTime();
        fileName = FileUtils.convertToSafeFileName(fileName);
        return new File(directory, fileName + ".tmp");
    }

    public static String getReadableFileSize(double sizeInBytes) {
        String sizeString;
        String units = BYTE_LABEL;
        if (sizeInBytes < 1024.0) {
            sizeString = "" + Math.round(sizeInBytes);
        } else {
            units = KILO_BYTE_LABEL;
            double sizeInKb = sizeInBytes / 1024.0;
            if (sizeInKb < 1024.0) {
                sizeString = "" + Math.round(sizeInKb);
            } else {
                units = MEGA_BYTE_LABEL;
                double sizeInMb = sizeInKb / 1024.0;
                if (sizeInMb < 1024.0) {
                    sizeString = "" + FileUtils.roundAndFormat(sizeInMb, 10.0f);
                } else {
                    units = GIGA_BYTE_LABEL;
                    double sizeInGb = sizeInMb / 1024.0;
                    sizeString = "" + FileUtils.roundAndFormat(sizeInGb, 10.0f);
                }
            }
        }
        return sizeString + " " + units;
    }

    private static String roundAndFormat(double size, float maxShowDecimals) {
        if (size < (double)maxShowDecimals) {
            return new DecimalFormat("#.0").format(size);
        }
        return "" + Math.round(size);
    }
}

