package uk.ac.warwick.util.ais.core.httpclient;

import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.junit.Test;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class AisHttpAsyncClientBaseTest {

    private static final String TEST_METHOD = "GET";
    private final AisHttpRequest request = new AisHttpRequest.Builder().path("path").build();

    @Test
    public void sendRequestAsync_buildRequestFailure_stopExecuteRequest() throws Exception {
        HttpRequestBuilder<HttpUriRequest> requestBuilder = (method, req) -> {
            throw new RuntimeException("Unable to build HttpUriRequest");
        };
        AisHttpResponseHandler<HttpResponse> responseHandler = mock(AisHttpResponseHandler.class);
        AtomicBoolean isExecuted = new AtomicBoolean(false);
        HttpRequestExecutor<HttpUriRequest, HttpResponse> executor = req -> {
            isExecuted.set(true);
            return CompletableFuture.completedFuture(mock(HttpResponse.class));
        };

        AisHttpAsyncClientBase<HttpUriRequest, HttpResponse> httpAsyncClient = new AisHttpAsyncClientBase<>(requestBuilder, responseHandler, executor);

        try {
            httpAsyncClient.sendRequestAsync(TEST_METHOD, request, new TypeReference<String>() {}).get();
            fail("Expected exception not thrown.");
        } catch (Exception ex) {
            assertTrue(ex instanceof RuntimeException);
            assertEquals("Unable to build HttpUriRequest", ex.getMessage());
            assertFalse(isExecuted.get()); // Ensure that the request is not executed
        }
    }

    @Test
    public void sendRequestAsync_executeRequestFailure_throwException() {
        HttpRequestBuilder<HttpUriRequest> requestBuilder = (method, req) -> mock(HttpUriRequest.class);
        AisHttpResponseHandler<HttpResponse> responseHandler = mock(AisHttpResponseHandler.class);
        when(responseHandler.handleError(any())).thenReturn(
                new AisHttpException(
                        AisHttpException.ErrorType.OTHERS,
                        "Execute Request Failed",
                        new RuntimeException("Execute Request Failed")
                )
        );

        AtomicBoolean isExecuted = new AtomicBoolean(false);
        HttpRequestExecutor<HttpUriRequest, HttpResponse> executor = req -> {
            isExecuted.set(true);
            CompletableFuture<HttpResponse> completableFuture = new CompletableFuture<>();
            completableFuture.completeExceptionally(new RuntimeException("Execute Request Failed"));
            return completableFuture;
        };

        AisHttpAsyncClientBase<HttpUriRequest, HttpResponse> httpAsyncClient = new AisHttpAsyncClientBase<>(requestBuilder, responseHandler, executor);

        try {
            httpAsyncClient.sendRequestAsync(TEST_METHOD, request, new TypeReference<String>() {}).get();
            fail("Expected exception not thrown.");
        } catch (Exception ex) {
            assertTrue(ex.getCause() instanceof AisHttpException);
            assertEquals("(errorType=OTHERS,message=Execute Request Failed)", ex.getCause().getMessage());
            assertTrue(isExecuted.get()); // Ensure that the request is executed
            verify(responseHandler).handleError(any());
        }
    }

    @Test
    public void sendRequestAsync_executeRequestSuccess_returnResponse() throws Exception {
        HttpRequestBuilder<HttpUriRequest> requestBuilder = (method, req) -> mock(HttpUriRequest.class);
        AisHttpResponseHandler<HttpResponse> responseHandler = mock(AisHttpResponseHandler.class);
        when(responseHandler.handleResponse(any(HttpResponse.class), any(TypeReference.class))).thenReturn("Response");

        AtomicBoolean isExecuted = new AtomicBoolean(false);
        HttpRequestExecutor<HttpUriRequest, HttpResponse> executor = req -> {
            isExecuted.set(true);
            return CompletableFuture.completedFuture(mock(HttpResponse.class));
        };

        AisHttpAsyncClientBase<HttpUriRequest, HttpResponse> httpAsyncClient = new AisHttpAsyncClientBase<>(requestBuilder, responseHandler, executor);

        String result = httpAsyncClient.sendRequestAsync(TEST_METHOD, request, new TypeReference<String>() {}).get();

        assertEquals("Response", result);
        assertTrue(isExecuted.get()); // Ensure that the request is executed
        verify(responseHandler).handleResponse(any(HttpResponse.class), any(TypeReference.class));
    }

    @Test
    public void sendRequest_timeoutException_throwException() throws Exception {
        HttpRequestBuilder<HttpUriRequest> requestBuilder = (method, req) -> mock(HttpUriRequest.class);
        AisHttpResponseHandler<HttpResponse> responseHandler = mock(AisHttpResponseHandler.class);
        AtomicBoolean isExecuted = new AtomicBoolean(false);
        HttpRequestExecutor<HttpUriRequest, HttpResponse> executor = req -> {
            isExecuted.set(true);
            return new CompletableFuture<>();
        };

        AisHttpAsyncClientBase<HttpUriRequest, HttpResponse> httpAsyncClient = new AisHttpAsyncClientBase<>(requestBuilder, responseHandler, executor);

        try {
            httpAsyncClient.sendRequest(TEST_METHOD, request, new TypeReference<String>() {}, 1, TimeUnit.MILLISECONDS);
            fail("Expected exception not thrown.");
        } catch (Exception ex) {
            assertTrue(ex instanceof AisHttpException);
            assertEquals("(errorType=CONNECTION_ERROR,message=Request was timed out.)", ex.getMessage());
            assertTrue(isExecuted.get()); // Ensure that the request is executed
            verifyZeroInteractions(responseHandler); // Ensure that the response handler is not called
        }
    }

    @Test
    public void sendRequest_aisHttpException_throwException() throws Exception {
        HttpRequestBuilder<HttpUriRequest> requestBuilder = (method, req) -> mock(HttpUriRequest.class);
        AisHttpResponseHandler<HttpResponse> responseHandler = mock(AisHttpResponseHandler.class);
        when(responseHandler.handleError(any())).thenReturn(
                new AisHttpException(
                        AisHttpException.ErrorType.OTHERS,
                        "Execute Request Failed",
                        new RuntimeException("Execute Request Failed")
                )
        );

        AtomicBoolean isExecuted = new AtomicBoolean(false);
        HttpRequestExecutor<HttpUriRequest, HttpResponse> executor = req -> {
            isExecuted.set(true);
            CompletableFuture<HttpResponse> completableFuture = new CompletableFuture<>();
            completableFuture.completeExceptionally(new RuntimeException("Execute Request Failed"));
            return completableFuture;
        };

        AisHttpAsyncClientBase<HttpUriRequest, HttpResponse> httpAsyncClient = new AisHttpAsyncClientBase<>(requestBuilder, responseHandler, executor);

        try {
            httpAsyncClient.sendRequest(TEST_METHOD, request, new TypeReference<String>() {}, 1, TimeUnit.SECONDS);
            fail("Expected exception not thrown.");
        } catch (Exception ex) {
            assertTrue(ex instanceof AisHttpException);
            assertEquals("(errorType=OTHERS,message=Execute Request Failed)", ex.getMessage());
            assertTrue(isExecuted.get()); // Ensure that the request is executed
            verify(responseHandler).handleError(any());
        }
    }

    @Test
    public void sendRequest_buildRequestFailure_throwException() throws Exception {
        HttpRequestBuilder<HttpUriRequest> requestBuilder = (method, req) -> {
            throw new RuntimeException("Unable to build HttpUriRequest");
        };
        AisHttpResponseHandler<HttpResponse> responseHandler = mock(AisHttpResponseHandler.class);
        AtomicBoolean isExecuted = new AtomicBoolean(false);
        HttpRequestExecutor<HttpUriRequest, HttpResponse> executor = req -> {
            isExecuted.set(true);
            return CompletableFuture.completedFuture(mock(HttpResponse.class));
        };

        AisHttpAsyncClientBase<HttpUriRequest, HttpResponse> httpAsyncClient = new AisHttpAsyncClientBase<>(requestBuilder, responseHandler, executor);

        try {
            httpAsyncClient.sendRequest(TEST_METHOD, request, new TypeReference<String>() {}, 1, TimeUnit.SECONDS);
            fail("Expected exception not thrown.");
        } catch (Exception ex) {
            assertTrue(ex instanceof AisHttpException);
            assertEquals("(errorType=OTHERS,message=An error occurred while executing the request.)", ex.getMessage());
            assertFalse(isExecuted.get()); // Ensure that the request is not executed
        }
    }

}
