package uk.ac.warwick.util.ais.core.helpers;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.Test;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpAsyncClient;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import static org.junit.Assert.*;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class AisHttpRequestLoggerTest {

    private static final String GET_METHOD = "GET";
    private final ObjectMapper objectMapper =  new ObjectMapper();

    @Test
    public void sendRequestAsync_successful() throws Exception {
        AisHttpRequest request = new AisHttpRequest.Builder().path("path").body("body").build();
        CompletableFuture<String> future = new CompletableFuture<>();
        future.complete("response");

        AisHttpAsyncClient delegate = mock(AisHttpAsyncClient.class);
        when(delegate.sendRequestAsync(
                eq(GET_METHOD),
                any(AisHttpRequest.class),
                any(TypeReference.class))
        ).thenReturn(future);

        AisHttpRequestLogger aisHttpRequestLogger = new AisHttpRequestLogger(objectMapper, delegate);

        CompletableFuture<String> resultFuture = aisHttpRequestLogger.sendRequestAsync(GET_METHOD, request, new TypeReference<String>() {});

        assertEquals("response", resultFuture.get(1, TimeUnit.SECONDS));
    }

    @Test
    public void sendRequestAsync_exception() {
        AisHttpRequest request = new AisHttpRequest.Builder().path("path").body("body").build();
        CompletableFuture<String> future = new CompletableFuture<>();
        future.completeExceptionally(new RuntimeException("Exception"));

        AisHttpAsyncClient delegate = mock(AisHttpAsyncClient.class);
        when(delegate.sendRequestAsync(
                eq(GET_METHOD),
                any(AisHttpRequest.class),
                any(TypeReference.class))
        ).thenReturn(future);

        AisHttpRequestLogger aisHttpRequestLogger = new AisHttpRequestLogger(objectMapper, delegate);

        CompletableFuture<String> resultFuture = aisHttpRequestLogger.sendRequestAsync(GET_METHOD, request, new TypeReference<String>() {});

        try {
            resultFuture.get(1, TimeUnit.SECONDS);
            fail("Expected an exception to be thrown");
        } catch (Exception e) {
            assertTrue(e.getCause() instanceof RuntimeException);
            assertEquals("Exception", e.getCause().getMessage());
        }
    }

    @Test
    public void sendRequest_successful() throws Exception {
        AisHttpRequest request = new AisHttpRequest.Builder().path("path").body("body").build();
        AisHttpAsyncClient delegate = mock(AisHttpAsyncClient.class);
        when(delegate.sendRequest(
                eq(GET_METHOD),
                any(AisHttpRequest.class),
                any(TypeReference.class),
                eq(1L),
                eq(TimeUnit.SECONDS))
        ).thenReturn("response");

        AisHttpRequestLogger aisHttpRequestLogger = new AisHttpRequestLogger(objectMapper, delegate);

        String result = aisHttpRequestLogger.sendRequest(GET_METHOD, request, new TypeReference<String>() {}, 1, TimeUnit.SECONDS);

        assertEquals("response", result);
    }

    @Test
    public void sendRequest_exception() {
        AisHttpRequest request = new AisHttpRequest.Builder().path("path").body("body").build();

        AisHttpAsyncClient delegate = mock(AisHttpAsyncClient.class);
        when(delegate.sendRequest(
                eq(GET_METHOD),
                any(AisHttpRequest.class),
                any(TypeReference.class),
                eq(1L),
                eq(TimeUnit.SECONDS))
        ).thenThrow(new RuntimeException("Exception"));

        AisHttpRequestLogger aisHttpRequestLogger = new AisHttpRequestLogger(objectMapper, delegate);
        try {
            aisHttpRequestLogger.sendRequest(GET_METHOD, request, new TypeReference<String>() {}, 1, TimeUnit.SECONDS);
            fail("Expected an exception to be thrown");
        } catch (Exception e) {
            assertTrue(e instanceof RuntimeException);
            assertEquals("Exception", e.getMessage());
        }
    }
}
