package uk.ac.warwick.util.ais.core.apache;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;
import org.junit.Test;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;

import static org.junit.Assert.*;
import static org.mockito.Mockito.mock;

public class DefaultHttpRequestExecutorTest {

    @Test
    public void execute_success() {
        CloseableHttpAsyncClient httpAsyncClient = new CloseableHttpAsyncClient() {
            public void close() throws IOException {
                // do nothing
            }

            public boolean isRunning() {
                return true;
            }

            public void start() {
                // do nothing
            }
            @Override
            public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer,
                                         HttpAsyncResponseConsumer<T> responseConsumer,
                                         HttpContext context,
                                         FutureCallback<T> callback) {
                HttpResponse response = mock(HttpResponse.class);
                callback.completed((T) response);
                return CompletableFuture.completedFuture((T) response);
            }
        };

        DefaultHttpRequestExecutor executor = new DefaultHttpRequestExecutor(httpAsyncClient);
        CompletableFuture<HttpResponse> response = executor.execute(buildRequest());
        assertNotNull(response);
        assertTrue(response.isDone());
        assertNotNull(response.getNow(null));
    }

    @Test
    public void execute_failure() {
        CloseableHttpAsyncClient httpAsyncClient = new CloseableHttpAsyncClient() {
            public void close() throws IOException {
                // do nothing
            }

            public boolean isRunning() {
                return true;
            }

            public void start() {
                // do nothing
            }
            @Override
            public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer,
                                         HttpAsyncResponseConsumer<T> responseConsumer,
                                         HttpContext context,
                                         FutureCallback<T> callback) {
                callback.failed(new RuntimeException("Request failed"));
                return new CompletableFuture<>();
            }
        };

        DefaultHttpRequestExecutor executor = new DefaultHttpRequestExecutor(httpAsyncClient);
        CompletableFuture<HttpResponse> response = executor.execute(buildRequest());
        assertNotNull(response);
        assertTrue(response.isCompletedExceptionally());
        try {
            response.getNow(null);
        } catch (Exception e) {
            assertTrue(e.getCause() instanceof RuntimeException);
            assertEquals("Request failed", e.getCause().getMessage());
        }
    }

    @Test
    public void execute_cancelled() {
        CloseableHttpAsyncClient httpAsyncClient = new CloseableHttpAsyncClient() {
            public void close() throws IOException {
                // do nothing
            }

            public boolean isRunning() {
                return true;
            }

            public void start() {
                // do nothing
            }
            @Override
            public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer,
                                         HttpAsyncResponseConsumer<T> responseConsumer,
                                         HttpContext context,
                                         FutureCallback<T> callback) {
                callback.cancelled();
                return new CompletableFuture<>();
            }
        };

        DefaultHttpRequestExecutor executor = new DefaultHttpRequestExecutor(httpAsyncClient);
        CompletableFuture<HttpResponse> response = executor.execute(buildRequest());
        assertNotNull(response);
        assertTrue(response.isCompletedExceptionally());
        try {
            response.getNow(null);
        } catch (Exception e) {
            assertTrue(e.getCause() instanceof RuntimeException);
            assertEquals("Request was cancelled.", e.getCause().getMessage());
        }
    }

    private HttpUriRequest buildRequest() {
        return RequestBuilder.create("GET")
                .setUri("http://test.url/test/api")
                .build();
    }
}
