package uk.ac.warwick.util.ais.core.apache;

import com.fasterxml.jackson.databind.JsonNode;
import org.apache.http.HttpResponse;
import org.apache.http.ProtocolVersion;
import org.apache.http.entity.BasicHttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicHttpResponse;
import org.junit.Before;
import org.junit.Test;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class DefaultAisHttpResponseHandlerTest {

    private AisJsonConverter jsonConverter;
    private DefaultAisHttpResponseHandler handler;

    @Before
    public void setUp() {
        jsonConverter = mock(AisJsonConverter.class);
        handler = new DefaultAisHttpResponseHandler(jsonConverter);
    }

    @Test
    public void handleResponseEntity_validResponse_returnsJsonNode() throws IOException {
        String jsonString = "{\"key\":\"value\"}";
        HttpResponse successResponse = new BasicHttpResponse(
                new ProtocolVersion("HTTP", 1, 1),
                200,
                "OK"
        );
        successResponse.setEntity(createHttpEntity(jsonString));

        when(jsonConverter.toJsonNode(jsonString)).thenReturn(mock(JsonNode.class));

        JsonNode result = handler.handleResponseEntity(successResponse);
        assertNotNull(result);
    }

    @Test
    public void handleResponseEntity_entityIsNull_returnsEmptyArrayJsonNode() throws IOException {
        HttpResponse successResponse = new BasicHttpResponse(
                new ProtocolVersion("HTTP", 1, 1),
                200,
                "OK"
        );
        successResponse.setEntity(null);

        when(jsonConverter.toJsonNode(anyString())).thenReturn(mock(JsonNode.class));

        JsonNode result = handler.handleResponseEntity(successResponse);
        assertNotNull(result);
    }

    @Test
    public void handleResponseEntity_statusCodeIs3xx_throwAisHttpException() throws IOException {
        String jsonString = "{\"key\":\"value\"}";
        HttpResponse errorResponse = new BasicHttpResponse(
                new ProtocolVersion("HTTP", 1, 1),
                300,
                "Redirect"
        );
        errorResponse.setEntity(createHttpEntity(jsonString));

        try {
            handler.handleResponseEntity(errorResponse);
            fail("Expected AisHttpException to be thrown");
        } catch (Exception e) {
            assertTrue(e instanceof AisHttpException);
            assertEquals(AisHttpException.ErrorType.HTTP_STATUS_CODE_ERROR, ((AisHttpException) e).getErrorType());
            assertEquals(jsonString, ((AisHttpException) e).getResponseBody());
            verify(jsonConverter, never()).toJsonNode(jsonString);
        }
    }

    @Test
    public void handleResponseEntity_statusCodeIs4xxAndEntityIsNull_throwAisHttpException() throws IOException {
        HttpResponse errorResponse = new BasicHttpResponse(
                new ProtocolVersion("HTTP", 1, 1),
                400,
                "Bad Request"
        );
        errorResponse.setEntity(null);

        try {
            handler.handleResponseEntity(errorResponse);
            fail("Expected AisHttpException to be thrown");
        } catch (Exception e) {
            assertTrue(e instanceof AisHttpException);
            assertEquals(AisHttpException.ErrorType.HTTP_STATUS_CODE_ERROR, ((AisHttpException) e).getErrorType());
            assertNull(((AisHttpException) e).getResponseBody());
            verify(jsonConverter, never()).toJsonNode(anyString());
        }
    }

    private BasicHttpEntity createHttpEntity(String content) {
        InputStream inputStream = new ByteArrayInputStream(content.getBytes());
        BasicHttpEntity httpEntity = new BasicHttpEntity();
        httpEntity.setContent(inputStream);
        httpEntity.setContentLength(content.length());
        httpEntity.setContentType(ContentType.APPLICATION_JSON.toString());
        httpEntity.setContentEncoding(new BasicHeader("Content-Encoding", "UTF-8"));
        return httpEntity;
    }
}
