package uk.ac.warwick.util.ais.apim.stutalk.json;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.scala.DefaultScalaModule;
import org.junit.Test;
import scala.collection.immutable.Seq;
import uk.ac.warwick.util.ais.apim.stutalk.TestData;
import uk.ac.warwick.util.ais.core.json.PipelineStringDeserializer;
import uk.ac.warwick.util.ais.core.json.PipelineStringSerializer;
import uk.ac.warwick.util.ais.core.json.TransformerPipelineFactory;

import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.List;

import static org.junit.Assert.*;

public class StuTalkJsonConverterImplTest {

    private final ObjectMapper objectMapper = new ObjectMapper()
            .registerModule(new JavaTimeModule()) // Register the JavaTimeModule to handle Java 8 Date/Time API
            .registerModule(new DefaultScalaModule()) // Register the DefaultScalaModule to handle Scala classes
            .registerModule(new JodaModule()) // Register the JodaModule to handle Joda Time classes
            .registerModule(buildCustomStutalkModule())
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .enable(DeserializationFeature.ACCEPT_EMPTY_ARRAY_AS_NULL_OBJECT)
            .setDateFormat(new SimpleDateFormat("yyyy-MM-dd"));

    private final StuTalkJsonConverter converter = new StuTalkJsonConverterImpl(objectMapper);
    private final TypeReference<List<TestData>> typeReference = new TypeReference<List<TestData>>() {};

    private final String listTestData = "{\n" +
            "   \"EXCHANGE\": {\n" +
            "     \"SRA\": {\n" +
            "       \"SRA.CAMS\": [\n" +
            "         {\n" +
            "           \"AYR_CODE.SRA.CAMS\": \"24/25\",\n" +
            "           \"MAV_OCCUR.SRA.CAMS\": \"A\",\n" +
            "           \"MOD_CODE.SRA.CAMS\": \"EN107-30\",\n" +
            "           \"PSL_CODE.SRA.CAMS\": \"Y\",\n" +
            "           \"SPR_CODE.SRA.CAMS\": \"5514251/1\",\n" +
            "           \"SRA_RSEQ.SRA.CAMS\": \"001\"\n" +
            "         }\n" +
            "       ]\n" +
            "     }\n" +
            "   }\n" +
            "}";

    private final String objTestData = "{\n" +
            "    \"EXCHANGE\": {\n" +
            "      \"SRA\": {\n" +
            "        \"SRA.CAMS\": {\n" +
            "            \"AYR_CODE.SRA.CAMS\": \"24/25\",\n" +
            "            \"MAV_OCCUR.SRA.CAMS\": \"A\",\n" +
            "            \"MOD_CODE.SRA.CAMS\": \"EN107-30\",\n" +
            "            \"PSL_CODE.SRA.CAMS\": \"Y\",\n" +
            "            \"SPR_CODE.SRA.CAMS\": \"5514251/1\",\n" +
            "            \"SRA_RSEQ.SRA.CAMS\": \"001\"\n" +
            "          }\n" +
            "      }\n" +
            "    }\n" +
            "}";

    private SimpleModule buildCustomStutalkModule() {
        SimpleModule stutalkModule = new SimpleModule();
        stutalkModule.addSerializer(String.class, new PipelineStringSerializer(TransformerPipelineFactory.createSerializer(true)));
        stutalkModule.addDeserializer(String.class, new PipelineStringDeserializer(TransformerPipelineFactory.createDeserializer(true, true)));
        return stutalkModule;
    }

    @Test
    public void fromJsonNode_validJsonNode_returnsDeserializedObject() {
        JsonNode jsonNode = converter.toJsonNode(listTestData);
        List<TestData> result = converter.fromJsonNode(jsonNode, typeReference);

        assertNotNull(result);
        assertEquals(1, result.size());

        assertEquals("24/25", result.get(0).getAyrCode());
        assertEquals("A", result.get(0).getMavOccur());
        assertEquals("EN107-30", result.get(0).getModCode());
        assertEquals("Y", result.get(0).getPslCode());
        assertEquals("5514251/1", result.get(0).getSprCode());
        assertEquals("001", result.get(0).getSraRseq());
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonNode_nullJsonNode_throwsIllegalArgumentException() {
        converter.fromJsonNode(null, typeReference);
    }

    @Test
    public void fromJsonNode_forClass_success() {
        JsonNode jsonNode = converter.toJsonNode(objTestData);
        TestData result = converter.fromJsonNode(jsonNode, TestData.class);

        assertNotNull(result);
        assertEquals("24/25", result.getAyrCode());
        assertEquals("A", result.getMavOccur());
        assertEquals("EN107-30", result.getModCode());
        assertEquals("Y", result.getPslCode());
        assertEquals("5514251/1", result.getSprCode());
        assertEquals("001", result.getSraRseq());
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonNode_forClassWithNullJsonNode_throwsIllegalArgumentException() {
        converter.fromJsonNode(null, TestData.class);
    }

    @Test
    public void fromJsonString_validJsonString_returnsDeserializedObject() {
        List<TestData> result = converter.fromJsonString(listTestData, typeReference);

        assertNotNull(result);
        assertEquals(1, result.size());

        assertEquals("24/25", result.get(0).getAyrCode());
        assertEquals("A", result.get(0).getMavOccur());
        assertEquals("EN107-30", result.get(0).getModCode());
        assertEquals("Y", result.get(0).getPslCode());
        assertEquals("5514251/1", result.get(0).getSprCode());
        assertEquals("001", result.get(0).getSraRseq());
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonString_nullJsonString_throwsIllegalArgumentException() {
        converter.fromJsonString(null, typeReference);
    }

    @Test
    public void fromJsonString_forClassWithValidJsonString_returnsDeserializedObject() {
        TestData result = converter.fromJsonString(objTestData, TestData.class);

        assertNotNull(result);
        assertEquals("24/25", result.getAyrCode());
        assertEquals("A", result.getMavOccur());
        assertEquals("EN107-30", result.getModCode());
        assertEquals("Y", result.getPslCode());
        assertEquals("5514251/1", result.getSprCode());
        assertEquals("001", result.getSraRseq());
    }

    @Test
    public void fromJsonString_withNewLine_returnsDeserializedObject() {
        String newLineTestData = "{\n" +
                "    \"EXCHANGE\": {\n" +
                "      \"SRA\": {\n" +
                "        \"SRA.CAMS\": {\n" +
                "            \"AYR_CODE.SRA.CAMS\": \"24/25\",\n" +
                "            \"MAV_OCCUR.SRA.CAMS\": \"\",\n" +
                "            \"MOD_CODE.SRA.CAMS\": \"EN107\\n-30\",\n" +
                "            \"PSL_CODE.SRA.CAMS\": \"Y\",\n" +
                "            \"SPR_CODE.SRA.CAMS\": \"5514251/1\",\n" +
                "            \"SRA_RSEQ.SRA.CAMS\": \"001\"\n" +
                "          }\n" +
                "      }\n" +
                "    }\n" +
                "}";
        TestData result = converter.fromJsonString(newLineTestData, TestData.class);

        assertNotNull(result);
        assertEquals("24/25", result.getAyrCode());
        assertNull(result.getMavOccur());
        assertEquals("EN107\r\n-30", result.getModCode());
        assertEquals("Y", result.getPslCode());
        assertEquals("5514251/1", result.getSprCode());
        assertEquals("001", result.getSraRseq());
    }

    @Test(expected = IllegalArgumentException.class)
    public void fromJsonString_forClassWithNullJsonString_throwsIllegalArgumentException() {
        converter.fromJsonString(null, TestData.class);
    }

    @Test
    public void toJsonString_listObject_returnsJsonString() {
        String exp = "{\"EXCHANGE\":{\"SRA\":{\"SRA.CAMS\":[{\"AYR_CODE.SRA.CAMS\":\"24/25\",\"MAV_OCCUR.SRA.CAMS\":\"A\"," +
                "\"MOD_CODE.SRA.CAMS\":\"EN107-30\",\"PSL_CODE.SRA.CAMS\":\"Y\",\"SPR_CODE.SRA.CAMS\":\"5514251/1\",\"SRA_RSEQ.SRA.CAMS\":\"001\"}]}}}";
        TestData data = new TestData("24/25", "A", "EN107-30", "Y", "5514251/1", "001");
        String json = converter.toJsonString(Collections.singletonList(data));
        assertEquals(exp, json);
    }

    @Test
    public void toJsonString_singleObject_returnsJsonString() {
        String exp = "{\"EXCHANGE\":{\"SRA\":{\"SRA.CAMS\":{\"AYR_CODE.SRA.CAMS\":\"24/25\",\"MAV_OCCUR.SRA.CAMS\":\"A\"," +
                "\"MOD_CODE.SRA.CAMS\":\"EN107-30\",\"PSL_CODE.SRA.CAMS\":\"Y\",\"SPR_CODE.SRA.CAMS\":\"5514251/1\",\"SRA_RSEQ.SRA.CAMS\":\"001\"}}}}";
        TestData data = new TestData("24/25", "A", "EN107-30", "Y", "5514251/1", "001");
        String json = converter.toJsonString(data);
        assertEquals(exp, json);
    }

    @Test
    public void toJsonString_nullObject_returnsEmptyJsonObject() {
        String result = converter.toJsonString(null);
        assertEquals("{}", result);
    }

    @Test
    public void toJsonString_scalaCollections_returnsJsonString() {
        String exp = "{\"EXCHANGE\":{\"SRA\":{\"SRA.CAMS\":[{\"AYR_CODE.SRA.CAMS\":\"24/25\",\"MAV_OCCUR.SRA.CAMS\":\"A\"," +
                "\"MOD_CODE.SRA.CAMS\":\"EN107-30\",\"PSL_CODE.SRA.CAMS\":\"Y\",\"SPR_CODE.SRA.CAMS\":\"5514251/1\",\"SRA_RSEQ.SRA.CAMS\":\"001\"}]}}}";
        TestData data = new TestData("24/25", "A", "EN107-30", "Y", "5514251/1", "001");

        Seq<TestData> scalaSeq = scala.jdk.CollectionConverters.CollectionHasAsScala(Collections.singletonList(data)).asScala().toSeq();

        String json = converter.toJsonString(scalaSeq);
        assertEquals(exp, json);
    }

    @Test
    public void toJsonString_normalizeNewLine_returnsJsonString() {
        String exp = "{\"EXCHANGE\":{\"SRA\":{\"SRA.CAMS\":{\"AYR_CODE.SRA.CAMS\":\"24/25\",\"MAV_OCCUR.SRA.CAMS\":\"A\"," +
                "\"MOD_CODE.SRA.CAMS\":\"EN107-30\",\"PSL_CODE.SRA.CAMS\":\"Y\",\"SPR_CODE.SRA.CAMS\":\"5514251/1\",\"SRA_RSEQ.SRA.CAMS\":\"001\\n002\"}}}}";
        TestData data = new TestData("24/25", "A", "EN107-30", "Y", "5514251/1", "001\r\n002");
        String json = converter.toJsonString(data);
        assertEquals(exp, json);
    }
}
