package uk.ac.warwick.util.ais.apim.stutalk;

import org.junit.Test;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.collections.Pair;

import static org.junit.Assert.*;

public class StuTalkHttpRequestBuilderTest {

    @Test(expected = IllegalArgumentException.class)
    public void build_apiCodeIsNull_throwException() {
        new StuTalkHttpRequestBuilder()
                .correlationId("correlationId")
                .body("body")
                .build();
    }

    @Test
    public void build_getRequest_success() {
        StuTalkHttpRequestBuilder builder = new StuTalkHttpRequestBuilder()
                .correlationId("correlationId")
                .apiCode("apiCode")
                .addPathVariable("key", "value")
                .addPathVariable("key2", "value/2");

        AisHttpRequest request = builder.build();
        assertEquals("correlationId", request.getCorrelationId());
        assertNotNull(request.getRequestId());
        assertNull(request.getBody());
        assertEquals("apiCode/key/value/key2/value@@2", request.getPath());
        assertEquals(0, request.getQueryParams().size());
        assertEquals(0, request.getHeaders().size());
    }

    @Test
    public void build_postRequest_success() {
        StuTalkHttpRequestBuilder builder = new StuTalkHttpRequestBuilder()
                .apiCode("apiCode")
                .body("body");

        AisHttpRequest request = builder.build();

        assertNotNull(request.getRequestId());
        assertEquals(request.getRequestId(), request.getCorrelationId());
        assertEquals("body", request.getBody());
        assertEquals("apiCode", request.getPath());
        assertEquals(0, request.getQueryParams().size());
        assertEquals(0, request.getHeaders().size());
    }

    @Test
    public void build_withQueryParamAndHeader_success() {
        StuTalkHttpRequestBuilder builder = new StuTalkHttpRequestBuilder()
                .apiCode("apiCode")
                .addPathVariable("key", "value/1/test")
                .addQueryParam("query-1", "param-1")
                .addQueryParam("query-2", "param-2")
                .addHeader(new Pair<>("header-1", "header-1-value"))
                .addHeader(new Pair<>("header-2", "header-2-value"));

        AisHttpRequest request = builder.build();

        assertNotNull(request.getRequestId());
        assertEquals(request.getRequestId(), request.getCorrelationId());
        assertNull(request.getBody());
        assertEquals("apiCode/key/value@@1@@test", request.getPath());
        assertEquals(2, request.getQueryParams().size());
        assertEquals("query-1", request.getQueryParams().get(0).getLeft());
        assertEquals("param-1", request.getQueryParams().get(0).getRight());
        assertEquals("query-2", request.getQueryParams().get(1).getLeft());
        assertEquals("param-2", request.getQueryParams().get(1).getRight());
        assertEquals(2, request.getHeaders().size());
        assertEquals("header-1", request.getHeaders().get(0).getLeft());
        assertEquals("header-1-value", request.getHeaders().get(0).getRight());
        assertEquals("header-2", request.getHeaders().get(1).getLeft());
        assertEquals("header-2-value", request.getHeaders().get(1).getRight());
    }

}
