package uk.ac.warwick.util.ais.core.resilience;

import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.retry.Retry;

import java.util.function.Function;
import java.util.function.Supplier;

/**
 * This is a minimal version of
 * <a href="https://github.com/resilience4j/resilience4j/blob/master/resilience4j-all/src/main/java/io/github/resilience4j/decorators/Decorators.java">
 * io.github.resilience4j.decorators.Decorators
 * </a>
 * because I don't really want to add too many unnecessary libraries
 * since the Decorators class is only provided in the resilience4j-all package.
 * <p>
 * A Decorator builder which can be used to apply multiple decorators to a Supplier, Callable
 * Function, Runnable, CompletionStage or Consumer.
 * <p>
 * Decorators are applied in the order of the builder chain. For example, consider:
 *
 * <pre>{@code
 * Supplier<String> supplier = ResilienceDecorators
 *     .ofSupplier(() -> service.method())
 *     .withCircuitBreaker(ResilienceConfig.createCircuitBreaker("id"))
 *     .withRetry(ResilienceConfig.createRetry("id"))
 *     .decorate();
 * }</pre>
 *
 * This results in the following composition when executing the supplier: <br>
 * <pre>Retry(CircuitBreaker(Supplier))</pre>
 *
 * This means the Supplier is called first, then its result is handled by the CircuitBreaker and then Retry.
 * Each Decorator makes its own determination whether an exception represents a failure.
 */
public interface ResilienceDecorators {
    static <T> DecorateSupplier<T> ofSupplier(Supplier<T> supplier) {
        return new DecorateSupplier<>(supplier);
    }

    static <T, R> DecorateFunction<T, R> ofFunction(Function<T, R> function) {
        return new DecorateFunction<>(function);
    }

    class DecorateSupplier<T> {

        private Supplier<T> supplier;

        private DecorateSupplier(Supplier<T> supplier) {
            this.supplier = supplier;
        }

        public DecorateSupplier<T> withCircuitBreaker(CircuitBreaker circuitBreaker) {
            supplier = CircuitBreaker.decorateSupplier(circuitBreaker, supplier);
            return this;
        }

        public DecorateSupplier<T> withRetry(Retry retryContext) {
            supplier = Retry.decorateSupplier(retryContext, supplier);
            return this;
        }

        public Supplier<T> decorate() {
            return supplier;
        }

        public T get() {
            return supplier.get();
        }
    }

    class DecorateFunction<T, R> {

        private Function<T, R> function;

        private DecorateFunction(Function<T, R> function) {
            this.function = function;
        }

        public DecorateFunction<T, R> withCircuitBreaker(CircuitBreaker circuitBreaker) {
            function = CircuitBreaker.decorateFunction(circuitBreaker, function);
            return this;
        }

        public DecorateFunction<T, R> withRetry(Retry retryContext) {
            function = Retry.decorateFunction(retryContext, function);
            return this;
        }

        public Function<T, R> decorate() {
            return function;
        }

        public R apply(T t) {
            return function.apply(t);
        }
    }
}
