package uk.ac.warwick.util.ais.core.json;

import java.util.ArrayList;
import java.util.List;

/**
 * Factory class for creating configurable {@link StringTransformer} pipelines used in
 * Jackson serialization and deserialization.
 *
 * <p>This factory assembles {@link ChainedTransformer} instances by conditionally adding
 * transformation steps based on runtime configuration.</p>
 *
 * @see StringTransformer
 * @see ChainedTransformer
 * @see PipelineStringDeserializer
 * @see PipelineStringSerializer
 */
public class TransformerPipelineFactory {

    /**
     * Creates a transformer pipeline for deserialization.
     *
     * @param shouldConvertEmptyStringsToNull whether to convert empty strings to null
     * @param shouldNormalizeNewlines         whether to normalize newline characters to Windows-style (\r\n)
     * @return a composed {@link StringTransformer} to be used in deserialization
     */
    public static StringTransformer createDeserializer(boolean shouldConvertEmptyStringsToNull, boolean shouldNormalizeNewlines) {
        List<StringTransformer> transformSteps = new ArrayList<>();

        if (shouldConvertEmptyStringsToNull) {
            transformSteps.add(new EmptyStringToNullTransformer());
        }
        if (shouldNormalizeNewlines) {
            transformSteps.add(new NewlineDeserializerTransformer());
        }

        return new ChainedTransformer(transformSteps);
    }

    /**
     * Creates a transformer pipeline for serialization.
     *
     * @param shouldNormalizeNewlines whether to convert Windows-style newlines (\r\n) to Unix-style (\n)
     * @return a composed {@link StringTransformer} to be used in serialization
     */
    public static StringTransformer createSerializer(boolean shouldNormalizeNewlines) {
        List<StringTransformer> transformSteps = new ArrayList<>();

        if (shouldNormalizeNewlines) {
            transformSteps.add(new NewlineSerializerTransformer());
        }

        return new ChainedTransformer(transformSteps);
    }
}
