package uk.ac.warwick.util.ais.core.json;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;

import java.io.IOException;

/**
 * A custom Jackson {@link JsonDeserializer} for {@link String} values that delegates transformation
 * logic to a configurable {@link StringTransformer} pipeline.
 *
 * <p>This deserializer enables the composition of multiple reusable transformation steps—
 * such as converting empty strings to {@code null}, normalizing newline characters, trimming whitespace, etc.—
 * in a flexible and declarative way.</p>
 *
 * @see StringTransformer
 * @see ChainedTransformer
 * @see PipelineStringSerializer
 */
public class PipelineStringDeserializer extends JsonDeserializer<String> {

    private final StringTransformer transformer;

    public PipelineStringDeserializer(StringTransformer transformer) {
        this.transformer = transformer;
    }

    /**
     * Reads the raw JSON string value and applies the configured transformation pipeline.
     *
     * @param jsonParser the Jackson parser
     * @param context    deserialization context
     * @return the transformed string
     * @throws IOException if parsing fails
     */
    @Override
    public String deserialize(JsonParser jsonParser, DeserializationContext context) throws IOException {
        String input = jsonParser.getValueAsString();
        return transformer.transform(input);
    }
}
