package uk.ac.warwick.util.ais.core.httpclient;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.SocketException;

public abstract class AbstractAisHttpResponseHandler<R> implements AisHttpResponseHandler<R> {

    private static final String SUPPORTED_CONTENT_TYPE = "application/json";

    protected final AisJsonConverter jsonConverter;

    protected AbstractAisHttpResponseHandler(AisJsonConverter jsonConverter) {
        this.jsonConverter = jsonConverter;
    }

    /**
     * Handle the response entity and transform it into a JsonNode object.
     * So, you can deserialise it into the actual response in the handleResponseEntity method.
     *
     * @param response The response to process
     * @return  The response as a JsonNode
     * @throws IOException  If an error occurs while processing the response
     */
    protected abstract JsonNode handleResponseEntity(R response) throws IOException;

    @Override
    public <T> T handleResponse(R response, TypeReference<T> typeReference) {
        try {
            JsonNode entity = handleResponseEntity(response);
            return jsonConverter.fromJsonNode(entity, typeReference);
        } catch (IOException ex) {
            // The IOException that occurs during JSON processing is caught
            throw new AisHttpException(
                    AisHttpException.ErrorType.OTHERS,
                    "ENTITY_MAPPING_EXCEPTION",
                    ex);
        }
    }

    @Override
    public AisHttpException handleError(Throwable ex) {

        if (ex instanceof AisHttpException) {
            // If the error is already classified as AisHttpException, re-throw it.
            return (AisHttpException) ex;
        } else if (ex instanceof SocketException || ex instanceof InterruptedIOException) {
            // If any network error
            return new AisHttpException(
                    AisHttpException.ErrorType.CONNECTION_ERROR,
                    "Network exception occurred.",
                    ex);
        } else {
            // If any other error
            return new AisHttpException(
                    AisHttpException.ErrorType.OTHERS,
                    "An error occurred while communicating with AIS API Management Service.",
                    ex);
        }
    }

    /**
     * Handle the response entity and transform it into the actual response object.
     */
    protected JsonNode handleEntity(String bodyString, String contentType) {

        // We only accept JSON response
        if (SUPPORTED_CONTENT_TYPE.equalsIgnoreCase(contentType)) {
            return jsonConverter.toJsonNode(bodyString);
        } else {
            throw new AisHttpException(
                    AisHttpException.ErrorType.MALFORMED_RESPONSE_ERROR,
                    String.format("Content type %s is not supported.", contentType),
                    null, // No status code
                    bodyString
            );
        }
    }
}
