package uk.ac.warwick.util.ais.core.helpers;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpAsyncClient;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.core.StopWatch;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * This class is responsible for logging the request, response and benchmark info for the Http Request execution.
 * We use the decorator pattern here to add additional code for logging purposes without changing the base-AisHttpAsyncClient.
 * This makes the usage more flexible, and eliminates Tight coupling in the base-AisHttpAsyncClient.
 */
public final class AisHttpRequestLogger implements AisHttpAsyncClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(AisHttpRequestLogger.class);

    private final ObjectMapper objectMapper;
    private final AisHttpAsyncClient delegate;

    public AisHttpRequestLogger(ObjectMapper objectMapper, AisHttpAsyncClient delegate) {
        this.objectMapper = objectMapper;
        this.delegate = delegate;
    }

    @Override
    public <T> CompletableFuture<T> sendRequestAsync(String method, AisHttpRequest request, TypeReference<T> responseType) {

        // Log the request info before executing the request
        logRequestInfo(method, request);

        StopWatch stopWatch = new StopWatch();
        stopWatch.start(request.getRequestId());

        // Execute the request to get back a CompletableFuture
        CompletableFuture<T> future = this.delegate.sendRequestAsync(method, request, responseType);

        // adding a callback to the future to log the response info and benchmark info
        future.whenComplete((result, throwable) -> {
            long executionTime = stopWatch.stop();
            if (throwable != null) {
                // Log the exception info if the request failed with an exception
                logExceptionInfo(method, request, throwable);
            } else {
                // Log the response info if the request was successful
                logResponseInfo(method, request, result);
            }
            // Log the benchmark info for the request execution
            logBenchmarkInfo(method, request, executionTime);
        });

        return future;
    }

    @Override
    public <T> T sendRequest(String method, AisHttpRequest request, TypeReference<T> responseType, long timeout, TimeUnit unit)
            throws AisHttpException {

        // Log the request info before executing the request
        logRequestInfo(method, request);

        StopWatch stopWatch = new StopWatch();
        stopWatch.start(request.getRequestId());

        // Execute the request to get back the response
        try {
            T result = this.delegate.sendRequest(method, request, responseType, timeout, unit);

            // Log the response info if the request was successful
            logResponseInfo(method, request, result);
            return result;
        } catch (Exception ex) {
            // Log the exception info if the request failed with an exception
            logExceptionInfo(method, request, ex);
            throw ex;
        } finally {
            // Log the benchmark info for the request execution
            logBenchmarkInfo(method, request, stopWatch.stop());
        }
    }

    private void logRequestInfo(String method, AisHttpRequest request) {
        String requestBody = toJsonString(request.getBody());
        LOGGER.info("[{}] Sends request to >>> Method={}, path={}, requestBody={}",
                request.getRequestId(),
                method,
                request.getPath(),
                requestBody
        );
    }

    private void logResponseInfo(String method, AisHttpRequest request, Object result) {
        String responseBody = toJsonString(result);
        LOGGER.info("[{}] Received response from <<< method={}, path={}, responseBody={}",
                request.getRequestId(),
                method,
                request.getPath(),
                responseBody
        );
    }

    private void logBenchmarkInfo(String method, AisHttpRequest request, Long executionTime) {
        LOGGER.info("[{}] Request to method={}, path={} finished in {} ms.",
                request.getRequestId(),
                method,
                request.getPath(),
                executionTime
        );
    }

    private void logExceptionInfo(String method, AisHttpRequest request, Throwable ex) {
        LOGGER.error("[{}] An error occurred while communicating with method={}, path={}, Error={}.",
                request.getRequestId(),
                method,
                request.getPath(),
                ex.getMessage()
        );
    }

    private String toJsonString(Object object) {
        if (object == null) return "null";

        try {
            return objectMapper.writeValueAsString(object);
        } catch (Exception ex) {
            // Returns a fixed string to indicate that the JSON serialization failed.
            // This is for logging only so do not handle anything here.
            return "JSON serialization failed";
        }
    }
}
