package uk.ac.warwick.util.ais.auth;

import uk.ac.warwick.util.ais.auth.credentials.OAuth2ClientCredentials;
import uk.ac.warwick.util.ais.auth.exception.AuthenticationException;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsRequest;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsResult;
import uk.ac.warwick.util.ais.auth.model.OAuth2TokenFetchParameters;
import uk.ac.warwick.util.ais.auth.token.AccessToken;
import uk.ac.warwick.util.ais.auth.token.TokenCache;

import java.util.function.Function;

public class OAuth2ClientCredentialsAuthenticator implements Authenticator<ClientCredentialsRequest, ClientCredentialsResult> {

    private static final String SHARP = "#";
    private final OAuth2ClientCredentials credentials;
    private final Function<OAuth2TokenFetchParameters, AccessToken> tokenFetcher;
    private final TokenCache tokenCache;

    public OAuth2ClientCredentialsAuthenticator(OAuth2ClientCredentials credentials,
                                                Function<OAuth2TokenFetchParameters, AccessToken> tokenFetcher,
                                                TokenCache tokenCache) {
        this.credentials = credentials;
        this.tokenFetcher = tokenFetcher;
        this.tokenCache = tokenCache;
    }

    @Override
    public ClientCredentialsResult authenticate(ClientCredentialsRequest request) throws AuthenticationException {
        try {

            // If the request specifies to use the cache, try to get the token from the cache
            if (request.getSkipCache() == Boolean.FALSE) {
                String key = credentials.getClientId() + SHARP + request.getScope();
                AccessToken cachedToken = tokenCache.getAccessToken(key, () -> fetchToken(request));
                return new ClientCredentialsResult(request.getScope(), cachedToken);
            }

            // If the request specifies to skip the cache, fetch a new token
            AccessToken accessToken = fetchToken(request);
            return new ClientCredentialsResult(request.getScope(), accessToken);
        } catch (Exception e) {
            throw new AuthenticationException("Failed to authenticate with the Authorization Server.", e);
        }
    }

    private AccessToken fetchToken(ClientCredentialsRequest request) {
        OAuth2TokenFetchParameters parameters = new OAuth2TokenFetchParameters(credentials, request.getScope());
        return tokenFetcher.apply(parameters);
    }
}
