package uk.ac.warwick.util.ais.apim.stutalk;

import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.collections.Pair;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import static uk.ac.warwick.util.ais.core.utils.StuTalkEncodeUtils.SLASH_CHAR;
import static uk.ac.warwick.util.ais.core.utils.StuTalkEncodeUtils.encode;

/**
 * StuTalk APIs only supports querying data using path variables or body but does not support query parameters and headers.
 *
 * Here is an example of how the Path API would be built using path variables:
 * <pre>
 * Given the following information
 *  - API gateway URL: https://test-studentdata.warwick.ac.uk/urd/sits.urd/run/SIW_RWS/
 *  - ApiCode: ESB_MOD_SMR
 *  - List of path variables:
 *      SPR_CODE = '5500503/1'
 *      MOD_CODE = 'WM160-15'
 *      MAV_OCCUR = 'A'
 *      AYR_CODE = '22/23'
 *      PSL_CODE = 'Y'
 * </pre>
 * The constructed request endpoint would be (the slash character would be encoded to '@@'):
 * <pre>
 * https://test-studentdata.warwick.ac.uk/urd/sits.urd/run/SIW_RWS/ESB_MOD_SMR/SPR_CODE/5500503@@1/MOD_CODE/WM160-15/MAV_OCCUR/A/AYR_CODE/22@@23/PSL_CODE/Y
 * </pre>
 */
public final class StuTalkHttpRequestBuilder {
    private final String requestId = UUID.randomUUID().toString(); // Unique identifier for the request
    private String correlationId = requestId; // correlationId defaults to requestId
    private String apiCode;
    private Object body;
    private final List<Pair<String, String>> pathVariables = new ArrayList<>();
    private final List<Pair<String, String>> queryParams = new ArrayList<>();
    private final List<Pair<String, String>> headers = new ArrayList<>();

    public StuTalkHttpRequestBuilder correlationId(String correlationId) {
        this.correlationId = correlationId;
        return this;
    }

    public StuTalkHttpRequestBuilder apiCode(String apiCode) {
        this.apiCode = apiCode;
        return this;
    }

    public StuTalkHttpRequestBuilder body(Object body) {
        this.body = body;
        return this;
    }

    public StuTalkHttpRequestBuilder addQueryParam(String key, String value) {
        this.queryParams.add(new Pair<>(key, value));
        return this;
    }

    public StuTalkHttpRequestBuilder addPathVariable(String key, String value) {
        this.pathVariables.add(new Pair<>(key, value));
        return this;
    }

    public StuTalkHttpRequestBuilder addHeader(Pair<String, String> header) {
        this.headers.add(header);
        return this;
    }

    public AisHttpRequest build() {
        if (apiCode == null) {
            throw new IllegalArgumentException("API Code cannot be null.");
        }
        String apiPath = buildPath();

        return new AisHttpRequest(
            correlationId,
            requestId,
            apiPath,
            body,
            queryParams,
            headers
        );
    }

    private String buildPath() {
        StringBuilder path = new StringBuilder(apiCode);
        for (Pair<String, String> pathVariable : pathVariables) {
            path.append(SLASH_CHAR)
                .append(pathVariable.getLeft())
                .append(SLASH_CHAR)
                .append(encode(pathVariable.getRight()));
        }
        return path.toString();
    }
}
