package uk.ac.warwick.util.ais.apim.config;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.scala.DefaultScalaModule;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import uk.ac.warwick.util.ais.apim.rodb.RodbHttpClient;
import uk.ac.warwick.util.ais.apim.rodb.RodbHttpClientImpl;
import uk.ac.warwick.util.ais.apim.stutalk.StuTalkHttpClient;
import uk.ac.warwick.util.ais.apim.stutalk.StuTalkHttpClientImpl;
import uk.ac.warwick.util.ais.apim.stutalk.json.StuTalkJsonConverterImpl;
import uk.ac.warwick.util.ais.auth.Authenticator;
import uk.ac.warwick.util.ais.auth.OAuth2ClientCredentialsAuthenticator;
import uk.ac.warwick.util.ais.auth.credentials.OAuth2ClientCredentials;
import uk.ac.warwick.util.ais.auth.exception.TokenFetchException;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsRequest;
import uk.ac.warwick.util.ais.auth.model.ClientCredentialsResult;
import uk.ac.warwick.util.ais.auth.model.OAuth2TokenFetchParameters;
import uk.ac.warwick.util.ais.auth.token.AccessToken;
import uk.ac.warwick.util.ais.auth.token.DefaultOAuth2TokenFetcher;
import uk.ac.warwick.util.ais.auth.token.TokenCache;
import uk.ac.warwick.util.ais.core.apache.DefaultAisHttpResponseHandler;
import uk.ac.warwick.util.ais.core.apache.DefaultHttpRequestBuilder;
import uk.ac.warwick.util.ais.core.apache.DefaultHttpRequestExecutor;
import uk.ac.warwick.util.ais.core.helpers.AisHttpRequestAuthorizer;
import uk.ac.warwick.util.ais.core.helpers.AisHttpRequestLogger;
import uk.ac.warwick.util.ais.core.httpclient.*;
import uk.ac.warwick.util.ais.core.json.*;
import uk.ac.warwick.util.ais.core.properties.AisApimProperties;
import uk.ac.warwick.util.ais.core.resilience.ResilienceConfig;
import uk.ac.warwick.util.ais.core.resilience.ResilienceDecorators;

import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

/**
 * The AisConfiguration class is responsible for configuring the AIS module in Spring.
 * It provides the necessary beans for the AIS module to function correctly.
 */
@Configuration
public class AisSpringConfiguration {

    /** The Connection Timeout determines the timeout in milliseconds until a connection is established. */
    private static final int CONNECT_TIME_OUT = 5000;
    /** The Socket Timeout defines the socket timeout in milliseconds. */
    private static final int SOCKET_TIME_OUT = 60000;
    /** The Connection Request Timeout is the timeout in milliseconds used when requesting a connection from the connection manager. */
    private static final int CONNECTION_REQUEST_TIME_OUT = 5000;

    private static final String YYYY_MM_DD_DATE_PATTERN = "yyyy-MM-dd";

    /** These are all temporary server errors that can be retried. */
    private static final Set<Integer> RETRIABLE_STATUS_CODE = Collections.unmodifiableSet(
            new HashSet<>(Arrays.asList(
                    500, // Internal Server Error
                    502, // Bad Gateway
                    503, // Service Unavailable
                    504 // Gateway Timeout
            ))
    );

    @Bean(name = "aisApacheHttpAsyncClient")
    public CloseableHttpAsyncClient httpAsyncClient(
            @Value("${ais.apim.http.maxConnections:10}") Integer maxConnections) {

        CloseableHttpAsyncClient httpAsyncClient = HttpAsyncClients.custom()
                .setDefaultRequestConfig(RequestConfig.custom()
                        .setConnectTimeout(CONNECT_TIME_OUT)
                        .setSocketTimeout(SOCKET_TIME_OUT)
                        .setConnectionRequestTimeout(CONNECTION_REQUEST_TIME_OUT)
                        .build())
                .setMaxConnPerRoute(maxConnections)
                .setMaxConnTotal(2 * maxConnections)
                .build();

        httpAsyncClient.start(); // Start the client by default to make it ready to execute requests

        return httpAsyncClient;
    }

    @Bean(name = "stuTalkObjectMapper")
    public ObjectMapper stuTalkObjectMapper(
            @Value("${ais.apim.objectMapper.emptyStringToNull:true}") boolean emptyStringToNull,
            @Value("${ais.apim.objectMapper.normaliseNewline:false}") boolean normaliseNewline // only apply new line Deserializer and Serializer when property is set.
    ) {
        return new ObjectMapper()
                .registerModule(new JavaTimeModule()) // Register the JavaTimeModule to handle Java 8 Date/Time API
                .registerModule(new DefaultScalaModule()) // Register the DefaultScalaModule to handle Scala classes
                .registerModule(new JodaModule()) // Register the JodaModule to handle Joda Time classes
                .registerModule(buildCustomStutalkModule(emptyStringToNull, normaliseNewline))
                .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
                .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .enable(DeserializationFeature.ACCEPT_EMPTY_ARRAY_AS_NULL_OBJECT)
                .setDateFormat(new SimpleDateFormat(YYYY_MM_DD_DATE_PATTERN));
    }

    /**
     * Builds a custom Jackson module that applies a configurable pipeline of transformers
     * for both serialization and deserialization of String values.
     *
     * <p>This allows us to define reusable, composable transformation steps (e.g. trimming,
     * newline normalization, empty string handling) that can be enabled/disabled via configuration,
     * and applied consistently across the entire application.</p>
     *
     * @param shouldConvertEmptyStringsToNull whether to convert empty strings to null during deserialization
     * @param shouldNormalizeNewlines         whether to normalise newlines during both serialization and deserialization
     * @return a Jackson Module that registers the custom serializer and deserializer for String
     */
    private Module buildCustomStutalkModule(boolean shouldConvertEmptyStringsToNull, boolean shouldNormalizeNewlines) {
        SimpleModule stutalkModule = new SimpleModule();
        stutalkModule.addSerializer(
            String.class,
            new PipelineStringSerializer(TransformerPipelineFactory.createSerializer(shouldNormalizeNewlines))
        );
        stutalkModule.addDeserializer(
            String.class,
            new PipelineStringDeserializer(TransformerPipelineFactory.createDeserializer(shouldConvertEmptyStringsToNull, shouldNormalizeNewlines))
        );
        return stutalkModule;
    }

    @Bean(name = "stuTalkJsonConverter")
    public AisJsonConverter stuTalkJsonConverter(
            @Qualifier("stuTalkObjectMapper") ObjectMapper objectMapper) {
        return new StuTalkJsonConverterImpl(objectMapper);
    }

    @Bean(name = "aisHttpResponseHandler")
    public AisHttpResponseHandler<HttpResponse> aisHttpResponseHandler(
            @Qualifier("stuTalkJsonConverter") AisJsonConverter stuTalkJsonConverter) {
        return new DefaultAisHttpResponseHandler(stuTalkJsonConverter);
    }

    @Bean(name = "aisHttpRequestExecutor")
    public HttpRequestExecutor<HttpUriRequest, HttpResponse> aisHttpRequestExecutor(
            @Qualifier("aisApacheHttpAsyncClient") CloseableHttpAsyncClient httpAsyncClient) {
        return new DefaultHttpRequestExecutor(httpAsyncClient);
    }

    @Bean(name = "aisOAuth2ClientCredentials")
    public OAuth2ClientCredentials aisOAuth2ClientCredentials(
            @Value("${ais.apim.auth.oauth2.auth_url:}") String authority,
            @Value("${ais.apim.auth.oauth2.client_id:}") String clientId,
            @Value("${ais.apim.auth.oauth2.client_secret:}") String clientSecret) {

        return new OAuth2ClientCredentials(authority, clientId, clientSecret);
    }

    @Bean(name = "aisAuthTokenCache")
    public TokenCache aisAuthTokenCache() {
        return new TokenCache();
    }

    @Bean(name = "aisAuthTokenFetcher")
    public Function<OAuth2TokenFetchParameters, AccessToken> aisAuthTokenFetcher(
            @Qualifier("stuTalkObjectMapper") ObjectMapper objectMapper,
            @Qualifier("aisHttpRequestExecutor") HttpRequestExecutor<HttpUriRequest, HttpResponse> httpRequestExecutor) {

        /*
         * Decorate the request with Circuit Breaker and Retry
         * This results in the following composition when executing the supplier: <br>
         * Retry(CircuitBreaker(sendRequest(accessTokenRequest)))
         * This means the Supplier is called first, then its result is handled by the CircuitBreaker and then Retry.
         */
        return ResilienceDecorators.ofFunction(new DefaultOAuth2TokenFetcher(objectMapper, httpRequestExecutor::execute))
                .withCircuitBreaker(ResilienceConfig.createCircuitBreaker("aisAuthCircuitBreaker"))
                .withRetry(ResilienceConfig.createRetry("aisAuthRetry", this::isRetriable))
                .decorate();
    }

    @Bean(name = "aisAuthenticator")
    public Authenticator<ClientCredentialsRequest, ClientCredentialsResult> aisAuthenticator(
            @Qualifier("aisOAuth2ClientCredentials") OAuth2ClientCredentials credentials,
            @Qualifier("aisAuthTokenFetcher") Function<OAuth2TokenFetchParameters, AccessToken> tokenFetcher,
            @Qualifier("aisAuthTokenCache") TokenCache tokenCache) {
        return new OAuth2ClientCredentialsAuthenticator(credentials, tokenFetcher, tokenCache);
    }

    @Bean(name = "aisApimProperties")
    public AisApimProperties aisApimProperties(
            @Value("${ais.apim.headers.bapi-channel-id}") String bapiChannelId,
            @Value("${ais.apim.headers.bapi-app-id}") String bapiAppId,
            @Value("${ais.apim.base-url}") String baseUrl) {

        return new AisApimProperties() {
            @Override
            public String getBapiChannelId() {
                return bapiChannelId;
            }

            @Override
            public String getBapiAppId() {
                return bapiAppId;
            }

            @Override
            public String getBaseUrl() {
                return baseUrl;
            }
        };
    }

    @Bean(name = "aisHttpRequestBuilder")
    public HttpRequestBuilder<HttpUriRequest> aisHttpRequestBuilder(
            @Qualifier("stuTalkJsonConverter") AisJsonConverter jsonConverter,
            @Qualifier("aisApimProperties") AisApimProperties aisApimProperties) {
        return new DefaultHttpRequestBuilder(jsonConverter, aisApimProperties);
    }

    @Bean(name = "aisAuthHttpRequestBuilder")
    public HttpRequestBuilder<HttpUriRequest> aisAuthHttpRequestBuilder(
            @Value("${ais.apim.auth.oauth2.scope:}") String defaultScope,
            @Qualifier("aisAuthenticator") Authenticator<ClientCredentialsRequest, ClientCredentialsResult> authenticator,
            @Qualifier("aisHttpRequestBuilder") HttpRequestBuilder<HttpUriRequest> requestBuilder) {
        return new AisHttpRequestAuthorizer<>(defaultScope, authenticator, requestBuilder);
    }

    @Bean(name = "aisHttpAsyncClient")
    public AisHttpAsyncClient aisHttpAsyncClient(
            @Qualifier("aisAuthHttpRequestBuilder") HttpRequestBuilder<HttpUriRequest> aisAuthHttpRequestBuilder,
            @Qualifier("aisHttpResponseHandler") AisHttpResponseHandler<HttpResponse> responseHandler,
            @Qualifier("aisHttpRequestExecutor") HttpRequestExecutor<HttpUriRequest, HttpResponse> httpRequestExecutor,
            @Qualifier("stuTalkObjectMapper") ObjectMapper objectMapper) {

        AisHttpAsyncClient delegate = new AisHttpAsyncClientBase<>(aisAuthHttpRequestBuilder, responseHandler, httpRequestExecutor);

        // Wrap the delegate with a logger to log the request and response info. Logging will be enabled by default.
        // You can disable it or create your own logger by implementing the AisHttpAsyncClient interface
        // then override this Spring bean with a @Primary annotation and same bean name.
        return new AisHttpRequestLogger(objectMapper, delegate);
    }

    @Bean(name = "stuTalkHttpClient")
    public StuTalkHttpClient stuTalkHttpClient(
            @Qualifier("aisHttpAsyncClient") AisHttpAsyncClient aisHttpAsyncClient) {
        return new StuTalkHttpClientImpl(aisHttpAsyncClient);
    }

    @Bean(name = "rodbHttpClient")
    public RodbHttpClient rodbHttpClient(
            @Qualifier("aisHttpAsyncClient") AisHttpAsyncClient aisHttpAsyncClient) {
        return new RodbHttpClientImpl(aisHttpAsyncClient);
    }

    private boolean isRetriable(Throwable ex) {
        // Do not retry if the exception is un-known
        if (!(ex instanceof TokenFetchException)) {
            return false;
        }
        TokenFetchException tokenEx = (TokenFetchException) ex;
        int statusCode = tokenEx.getStatusCode();
        Throwable cause = tokenEx.getCause();

        // Retry if the status code is one of the retriable status codes
        // or if the cause is a SocketException, SocketTimeoutException or TimeoutException
        return RETRIABLE_STATUS_CODE.contains(statusCode) ||
                cause instanceof SocketException ||
                cause instanceof SocketTimeoutException ||
                cause instanceof TimeoutException;
    }
}
