package uk.ac.warwick.util.ais.core.apache;

import com.google.common.collect.Lists;
import org.apache.http.client.methods.HttpUriRequest;
import org.junit.Before;
import org.junit.Test;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;
import uk.ac.warwick.util.ais.core.properties.AisApimProperties;
import uk.ac.warwick.util.collections.Pair;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class DefaultHttpRequestBuilderTest {

    private final AisApimProperties properties = new AisApimProperties() {
        public String getBapiChannelId() {
            return "bapiChannelId";
        }
        public String getBapiAppId() {
            return "bapiAppId";
        }
        public String getBaseUrl() {
            return "https://base.url/";
        }
    };

    private AisJsonConverter jsonConverter;

    @Before
    public void setup() {
        jsonConverter = mock(AisJsonConverter.class);
    }

    @Test
    public void buildHttpRequest_getRequest_success() {
        AisHttpRequest request = new AisHttpRequest(
                "correlationId",
                "requestId",
                "/test/api/key1/value1<GOLD>*/key2/value2",
                null,
                Lists.newArrayList(
                        new Pair<>("param-1", "query-1"),
                        new Pair<>("param-2", "query-2")
                ),
                Lists.newArrayList(
                        new Pair<>("header-1", "value1"),
                        new Pair<>("header-2", "value2")
                )
        );
        DefaultHttpRequestBuilder builder = new DefaultHttpRequestBuilder(jsonConverter, properties);
        HttpUriRequest result = builder.buildHttpRequest("GET", request);

        assertNotNull(result);
        assertEquals("GET", result.getMethod());
        assertEquals("https://base.url/test/api/key1/value1%3CGOLD%3E*/key2/value2?param-1=query-1&param-2=query-2", result.getURI().toString());
        assertEquals("header-1", result.getFirstHeader("header-1").getName());
        assertEquals("value1", result.getFirstHeader("header-1").getValue());
        assertEquals("header-2", result.getFirstHeader("header-2").getName());
        assertEquals("value2", result.getFirstHeader("header-2").getValue());
        assertEquals("bapiChannelId", result.getFirstHeader("bapi-channel-id").getValue());
        assertEquals("bapiAppId", result.getFirstHeader("bapi-app-id").getValue());
        assertEquals("correlationId", result.getFirstHeader("bapi-correlation-id").getValue());
        assertEquals("requestId", result.getFirstHeader("bapi-request-id").getValue());
    }

    @Test
    public void buildHttpRequest_postRequest_success() {
        AisHttpRequest request = new AisHttpRequest(
                null,
                null,
                "/test/api",
                "body",
                null,
                null
        );
        DefaultHttpRequestBuilder builder = new DefaultHttpRequestBuilder(jsonConverter, properties);
        when(jsonConverter.toJsonString("body")).thenReturn("body");

        HttpUriRequest result = builder.buildHttpRequest("POST", request);

        assertNotNull(result);
        assertEquals("POST", result.getMethod());
        assertEquals("https://base.url/test/api", result.getURI().toString());
        assertEquals("bapiChannelId", result.getFirstHeader("bapi-channel-id").getValue());
        assertEquals("bapiAppId", result.getFirstHeader("bapi-app-id").getValue());
        assertNotNull(result.getFirstHeader("bapi-correlation-id").getValue()); // correlationId is generated
        assertNotNull("requestId", result.getFirstHeader("bapi-request-id").getValue()); // requestId is generated
    }

    @Test
    public void buildHttpRequest_withBasePath_success() {
        AisHttpRequest request = new AisHttpRequest.Builder()
                .path("/test/api/key1/value1<GOLD>|value2") // <GOLD>| is the wildcard for OR condition supported by StuTalk
                .correlationId("correlationId")
                .addQueryParam(new Pair<>("param-1", "pValue-1"))
                .build();

        DefaultHttpRequestBuilder builder = new DefaultHttpRequestBuilder(jsonConverter, new AisApimProperties() {
            public String getBapiChannelId() {
                return "bapiChannelId";
            }
            public String getBapiAppId() {
                return "bapiAppId";
            }
            public String getBaseUrl() {
                return "https://base.url/basepath/";
            }
        });

        HttpUriRequest result = builder.buildHttpRequest("GET", request);

        assertNotNull(result);
        assertEquals("GET", result.getMethod());
        assertEquals("https://base.url/basepath/test/api/key1/value1%3CGOLD%3E%7Cvalue2?param-1=pValue-1", result.getURI().toString());
        assertEquals("bapiChannelId", result.getFirstHeader("bapi-channel-id").getValue());
        assertEquals("bapiAppId", result.getFirstHeader("bapi-app-id").getValue());
        assertEquals("correlationId", result.getFirstHeader("bapi-correlation-id").getValue());
    }
}
