package uk.ac.warwick.util.ais.core.utils;

/**
 * This class is used to encode and decode string values for StuTalk API request.
 */
public final class StuTalkEncodeUtils {

    // SITS has specific characters that function as operators.
    // These characters are not supported with API calls and need to be substituted for specific strings.

    public static final String SLASH_CHAR = "/";

    public static final String ENCODED_SLASH = "@@";

    public static final String ALL_MATCHING = "<GOLD>*";

    /**
     * StuTalk Wildcard to build NULL match condition.<br>
     * In some explicit cases, We need to retrieve records that have a specific column that is NULL value
     */
    public static final String NULL_MATCHING = "<GOLD>NULL";

    /** StuTalk Wildcard to build OR match condition. */
    public static final String OR_MATCHING = "<GOLD>|";

    /** StuTalk Wildcard to build Not equals match condition. */
    public static final String NE_MATCHING = "<GOLD>NE";

    /** StuTalk Wildcard to build greater than or equals match condition. */
    public static final String GE_MATCHING = "<GOLD>GE";

    /** StuTalk Wildcard to build greater than match condition. */
    public static final String G_MATCHING = "<GOLD>G";

    /** StuTalk Wildcard to build less than or equals match condition. */
    public static final String LE_MATCHING = "<GOLD>LE";

    /** StuTalk Wildcard to build less than match condition. */
    public static final String L_MATCHING = "<GOLD>L";

    private static final String NULL_ERROR_MESSAGE = "Value cannot be null. Please use NULL_MATCHING instead.";

    private StuTalkEncodeUtils() {
        throw new UnsupportedOperationException("Utility class cannot be instantiated directly.");
    }

    /**
     * This method is used to make input string value be able to use Wildcard as partial matching.
     * It is similar to LIKE operator SQL.
     * For example, we want to get CAM_SAS records with sql condition: SPR_CODE LIKE '5553778%'
     * Then the partial matching string for API request will be: SPR_CODE/5553778<GOLD*>
     *
     * @param value given string value
     * @return the string value with partial matching
     */
    public static String startsWith(String value) {
        if (value == null) {
            throw new IllegalArgumentException(NULL_ERROR_MESSAGE);
        }
        return encode(value).concat(ALL_MATCHING);
    }

    /**
     * This method is used to make input string value be able to use Wildcard as greater than or equal matching.
     * It is similar to >= operator SQL.
     * Example: SPR_CODE/<GOLD>GE5553778
     *
     * @param value inputted string value
     * @return the string value with greater than or equal matching
     */
    public static String greaterThanOrEqualTo(String value) {
        return concat(GE_MATCHING, value);
    }

    public static String greaterThan(String value) {
        return concat(G_MATCHING, value);
    }

    /**
     * This method is used to make inputted string value be able to use Wildcard as less or equal matching
     * Example: SPR_CODE/<GOLD>LE5553778
     *
     * @param value inputted string value
     * @return the string value with less or equal matching
     */
    public static String lessThanOrEqualTo(String value) {
        return concat(LE_MATCHING, value);
    }

    public static String lessThan(String value) {
        return concat(L_MATCHING, value);
    }

    public static String notEqualTo(String value) {
        return concat(NE_MATCHING, value);
    }

    public static String isNull() {
        return NULL_MATCHING;
    }

    public static String isAny() {
        return ALL_MATCHING;
    }

    public static String either(String a, String b) {
        return concat(concat(a, OR_MATCHING), b);
    }

    public static String encode(String value) {
        if (value == null) {
            // enforce the use of NULL_MATCHING instead of null
            throw new IllegalArgumentException(NULL_ERROR_MESSAGE);
        }
        return value.replace(SLASH_CHAR, ENCODED_SLASH); // encode SLASH (/) character
    }

    public static String decode(String value) {
        if (value == null)  return null; // do nothing if value is null
        return value.replace(ENCODED_SLASH, SLASH_CHAR); // decode SLASH (/) character
    }

    private static String concat(String a, String b) {
        if (a == null || b == null) {
            throw new IllegalArgumentException(NULL_ERROR_MESSAGE);
        }

        return a.concat(b);
    }

}
