package uk.ac.warwick.util.ais.core.resilience;

import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.circuitbreaker.CircuitBreakerConfig;
import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;

import java.time.Duration;
import java.util.function.Predicate;

public final class ResilienceConfig {

    /**
     * The percentage of failures that will trip the circuit breaker
     * (e.g., 50% means half of the calls must fail to open the circuit).
     */
    private static final float CB_FAILURE_RATE_THRESHOLD = 50;

    /** The time the circuit breaker will stay open before transitioning to half-open state. (60 seconds). */
    private static final long CB_OPEN_STATE_DURATION = 60;

    /** The size of the sliding window used to record the outcome of calls. */
    private static final int CB_SLIDING_WINDOW_SIZE = 10;

    /** The number of calls allowed when the circuit breaker is in half-open state. */
    private static final int CB_HALF_OPEN_STATE_PERMITTED_CALLS = 2;

    /** The maximum number of times the operation will be retried before giving up */
    private static final int RETRY_MAX_ATTEMPTS = 3;

    /** The initial wait time between retry attempts (5 seconds) */
    private static final long RETRY_WAITING_INTERVAL = 5000;

    /** The factor by which the wait time increases after each retry (2 for doubling the wait time) */
    private static final double RETRY_WAITING_INTERVAL_MULTIPLIER = 2;

    private ResilienceConfig() {
        // Prevent instantiation
        throw new UnsupportedOperationException("This class should not be instantiated.");
    }

    /**
     * CircuitBreaker is used to prevent calls to a service that is currently failing by opening the circuit breaker
     * after a failure threshold is reached.
     * <p>
     * CircuitBreaker should be <code>stateful</code> and should be shared across all requests
     * so if several requests to the Authorization Server fail, the CircuitBreaker can open
     * and block further requests until a cooldown period is reached,
     * preventing overloading the Authorization Server
     *
     * @see <a href="https://resilience4j.readme.io/docs/circuitbreaker">CircuitBreaker</a>
     * @see <a href="https://resilience4j.readme.io/docs/circuitbreaker#create-and-configure-a-circuitbreaker">Create and configure a CircuitBreaker</a>
     */
    public static CircuitBreaker createCircuitBreaker(String name) {
        CircuitBreakerConfig config = CircuitBreakerConfig.custom()
                .failureRateThreshold(CB_FAILURE_RATE_THRESHOLD)
                .waitDurationInOpenState(Duration.ofSeconds(CB_OPEN_STATE_DURATION))
                .permittedNumberOfCallsInHalfOpenState(CB_HALF_OPEN_STATE_PERMITTED_CALLS)
                .slidingWindowSize(CB_SLIDING_WINDOW_SIZE)
                .build();
        return createCircuitBreaker(name, config);
    }

    public static CircuitBreaker createCircuitBreaker(String name, CircuitBreakerConfig config) {
        return CircuitBreaker.of(name, config);
    }

    /**
     * Create a noop CircuitBreaker that will never open.
     * @return a CircuitBreaker instance with a sliding window size of 1 and a failure rate threshold of 100%
     */
    public static CircuitBreaker noopCircuitBreaker() {
        CircuitBreakerConfig config = CircuitBreakerConfig.custom()
                .slidingWindowSize(1)
                .failureRateThreshold(100)
                .build();
        return createCircuitBreaker("noop", config);
    }

    /**
     * Retry is used to automatically retry a failed operation with a configurable backoff strategy.
     * <p>
     * Retry should be <code>stateless</code> and should be used for each request
     * so The exponential backoff will apply only to a single request.
     * After the retry attempts are exhausted, the next request will start fresh with no
     * relation to the previous backoff attempts.
     * <p>
     * This approach avoids excessively long delays between retries across different requests.<br>
     * It should be used to retry the request if the Authorization Server is
     * temporarily unavailable or the request fails due to network issues.<br>
     * It should not be used to retry the request if the request is invalid or the credentials are incorrect
     * (e.g. 4xx errors).
     *
     * @see <a href="https://resilience4j.readme.io/docs/retry">Retry</a>
     */
    public static Retry createRetry(String name, Predicate<Throwable> retryOnException) {
        RetryConfig config = RetryConfig.custom()
                .maxAttempts(RETRY_MAX_ATTEMPTS)
                .intervalFunction(
                        IntervalFunction.ofExponentialBackoff(RETRY_WAITING_INTERVAL, RETRY_WAITING_INTERVAL_MULTIPLIER)
                )
                .retryOnException(retryOnException)
                .build();
        return createRetry(name, config);
    }

    public static Retry createRetry(String name, RetryConfig config) {
        return Retry.of(name, config);
    }

    /**
     * This is mainly used if the request is idempotent and the client can safely retry the request,
     * or if the client can handle the failure gracefully without retrying the request.
     * @return  a Retry instance with no retry attempts
     */
    public static Retry noopRetry() {
        RetryConfig config = RetryConfig.custom()
                .maxAttempts(1) // no retry
                .build();
        return createRetry("noop", config);
    }
}
