package uk.ac.warwick.util.ais.core.httpclient;

import com.fasterxml.jackson.core.type.TypeReference;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;

import java.util.concurrent.*;

public class AisHttpAsyncClientBase<Req, Res> implements AisHttpAsyncClient {

    private final HttpRequestBuilder<Req> requestBuilder;
    private final AisHttpResponseHandler<Res> responseHandler;
    private final HttpRequestExecutor<Req, Res> executor;

    public AisHttpAsyncClientBase(HttpRequestBuilder<Req> requestBuilder,
                                  AisHttpResponseHandler<Res> responseHandler,
                                  HttpRequestExecutor<Req, Res> executor) {
        this.requestBuilder = requestBuilder;
        this.responseHandler = responseHandler;
        this.executor = executor;
    }

    @Override
    public <T> CompletableFuture<T> sendRequestAsync(String method, AisHttpRequest request, TypeReference<T> responseType) {

        Req httpRequest = requestBuilder.buildHttpRequest(method, request);

        // performs an asynchronous HTTP request to the specified URL.
        // It returns a CompletableFuture<HttpResponse> which will eventually complete with the response or an exception.
        CompletableFuture<Res> futureResponse = executor.execute(httpRequest);

        // thenApply is used to handle the successful completion of the CompletableFuture.
        // exceptionally is used to handle any exceptions that occur during the request execution.
        return futureResponse.thenApply(
                response -> responseHandler.handleResponse(response, responseType)
        ).exceptionally(ex -> {
            Throwable cause = ex;
            if (ex instanceof CompletionException) {
                // Unwrap the CompletionException to get the actual cause of the exception
                cause = ex.getCause();
            }
            throw responseHandler.handleError(cause);
        });
    }

    @Override
    public <T> T sendRequest(String method, AisHttpRequest request, TypeReference<T> responseType, long timeout, TimeUnit unit)
            throws AisHttpException {
        try {
            return sendRequestAsync(method, request, responseType).get(timeout, unit);
        } catch (Throwable ex) {
            throw handleSyncRequestException(ex);
        }
    }

    /**
     * All exception need to be mapped to AisHttpException.
     *
     * @param ex the exception
     * @return the AisHttpException
     */
    private AisHttpException handleSyncRequestException(Throwable ex) {
        Throwable cause = ex;
        if (cause instanceof InterruptedException) {
            Thread.currentThread().interrupt(); // Restore the interrupted status
        }
        if (cause instanceof TimeoutException) {
            return new AisHttpException(
                    AisHttpException.ErrorType.CONNECTION_ERROR,
                    "Request was timed out.",
                    ex);
        }
        if (ex instanceof ExecutionException) {
            cause = ex.getCause(); // Unwrap the ExecutionException to get the actual cause of the exception
        }

        if (cause instanceof AisHttpException) {
            // If the error is already classified as AisHttpException by the above responseHandler.handleError, re-throw it.
            return (AisHttpException) cause;
        }

        return new AisHttpException(
                AisHttpException.ErrorType.OTHERS,
                "An error occurred while executing the request.",
                cause);
    }
}
