package uk.ac.warwick.util.ais.core.exception;

import java.util.StringJoiner;

public class AisHttpException extends RuntimeException {

    private static final long serialVersionUID = -1193143672804622254L;

    /** Error type */
    private final ErrorType errorType;

    /** HTTP status code*/
    private final Integer statusCode;

    /** Response body */
    private final String responseBody;

    public AisHttpException(ErrorType errorType, String message, Integer statusCode, String responseBody) {

        super(createMessage(errorType, message, statusCode, responseBody));
        this.errorType = errorType;
        this.statusCode = statusCode;
        this.responseBody = responseBody;
    }

    public AisHttpException(ErrorType errorType, String message, Integer statusCode, String responseBody, Throwable cause) {
        this(errorType, message, statusCode, responseBody);
        initCause(cause);
    }

    public AisHttpException(ErrorType errorType, String message, Throwable cause) {

        super(createMessage(errorType, message));
        this.errorType = errorType;
        this.statusCode = null; // No status code
        this.responseBody = null; // No response body
        initCause(cause);
    }

    public ErrorType getErrorType() {
        return errorType;
    }

    public Integer getStatusCode() {
        return statusCode;
    }

    public String getResponseBody() {
        return responseBody;
    }

    /**
     * Create a friendly detailed message.
     *
     * @param errorType     error type
     * @param message       error message
     * @param statusCode    HTTP status code (null can be set)
     * @param responseBody  Response body (null can be set)
     * @return Detailed message
     */
    private static String createMessage(ErrorType errorType, String message, Integer statusCode, String responseBody) {

        StringJoiner joiner = new StringJoiner(",", "(", ")");

        joiner.add(String.format("errorType=%s", errorType));
        joiner.add(String.format("message=%s", message));
        joiner.add(String.format("statusCode=%03d", statusCode));
        joiner.add(String.format("responseBody=%s", responseBody));
        return joiner.toString();
    }

    /**
     * Create a friendly detailed message.
     *
     * @param errorType     error type
     * @param message       error message
     * @return Detailed message
     */
    private static String createMessage(ErrorType errorType, String message) {

        StringJoiner joiner = new StringJoiner(",", "(", ")");

        joiner.add(String.format("errorType=%s", errorType));
        joiner.add(String.format("message=%s", message));
        return joiner.toString();
    }

    public enum ErrorType {

        /**
         * Receives an error response from the AIS API, these errors will be classified based on the Http Status Code.
         *
         * <p>Typically, it could be one of the following:</p>
         * <p>3xx Redirection: Further action needs to be taken in order to complete the request.</p>
         * <p>4xx Client Error: The request contains bad syntax or cannot be fulfilled.</p>
         * <p>5xx Server Error: The server failed to fulfill an apparently valid request.</p>
         */
        HTTP_STATUS_CODE_ERROR,

        /**
         * Receives a Malformed response from the AIS API such as invalid JSON or Unsupported content type, etc.
         * This error is usually used in case we receive a 2xx response code but the response body is not in the expected format.
         * or the content type is not supported.
         */
        MALFORMED_RESPONSE_ERROR,

        /**
         * Connection Error represents all Network/IO failures. This always comes with an exception cause.
         * These failures could be:
         *
         * <p>Timeouts: These errors occur when Tabula cannot connect to the AIS API Service or when the Service does not respond within a specified amount of time.
         * Such as: ConnectTimeoutException, SocketTimeoutException.</p>
         * <p>Network error: When Tabula cannot establish connection to the AIS API Service, it is usually because the service is down or unavailable.
         * Such as: UnknownHostException, NoRouteToHostException, ConnectException.</p>
         * <p>SSL/TLS errors: These errors occur when there is a problem with the SSL/TLS handshake between Tabula and the AIS API Service.
         * Such as: SSLHandshakeException, SSLPeerUnverifiedException, SSLException.</p>
         */
        CONNECTION_ERROR,

        /** Others */
        OTHERS
    }
}
