package uk.ac.warwick.util.ais.core.apache;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import uk.ac.warwick.util.ais.core.httpclient.HttpRequestExecutor;

import java.util.concurrent.CompletableFuture;

/**
 * This DefaultHttpRequestExecutor is an implementation of HttpRequestExecutor interface
 * which is mainly used to work with Apache HttpClient library.
 */
public final class DefaultHttpRequestExecutor implements HttpRequestExecutor<HttpUriRequest, HttpResponse> {

    private final CloseableHttpAsyncClient httpAsyncClient;

    public DefaultHttpRequestExecutor(CloseableHttpAsyncClient httpAsyncClient) {
        this.httpAsyncClient = httpAsyncClient;
    }

    @Override
    public CompletableFuture<HttpResponse> execute(HttpUriRequest request) {

        CompletableFuture<HttpResponse> futureResponse = new CompletableFuture<>();

        /*
         * CloseableHttpAsyncClient.sendRequest is a Java method that will return a Java Future object.
         * Java Future is a poor type, so we are using CompletableFuture which extends to the Future.
         * I believe this will be completely asynchronous and non-blocking.
         */
        httpAsyncClient.execute(request, new FutureCallback<HttpResponse>() {
            @Override
            public void completed(HttpResponse result) {
                futureResponse.complete(result);
            }
            @Override
            public void failed(Exception ex) {
                futureResponse.completeExceptionally(ex);
            }
            @Override
            public void cancelled() {
                futureResponse.completeExceptionally(new RuntimeException("Request was cancelled."));
            }
        });

        return futureResponse;
    }
}
