package uk.ac.warwick.util.ais.core.apache;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.entity.EntityBuilder;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import uk.ac.warwick.util.ais.core.httpclient.AisHttpRequest;
import uk.ac.warwick.util.ais.core.httpclient.HttpRequestBuilder;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;
import uk.ac.warwick.util.ais.core.properties.AisApimProperties;
import uk.ac.warwick.util.collections.Pair;

import java.net.URISyntaxException;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;

import static uk.ac.warwick.util.ais.core.helpers.AisApimConstants.*;

/**
 * Default implementation of {@link HttpRequestBuilder} that creates a new {@link HttpUriRequest} instance.
 */
public class DefaultHttpRequestBuilder implements HttpRequestBuilder<HttpUriRequest> {

    private final AisJsonConverter jsonConverter;
    private final AisApimProperties properties;

    public DefaultHttpRequestBuilder(AisJsonConverter jsonConverter, AisApimProperties properties) {
        this.jsonConverter = jsonConverter;
        this.properties = properties;
    }

    @Override
    public HttpUriRequest buildHttpRequest(String method, AisHttpRequest request) {

        // Create a new RequestBuilder instance with the specified method and URI
        RequestBuilder builder = RequestBuilder.create(method)
                .setUri(getUri(properties.getBaseUrl(), request.getPath()));

        // Add body to request
        if (request.getBody() != null) {
            builder.setEntity(
                    EntityBuilder.create()
                            .setContentType(ContentType.APPLICATION_JSON)
                            .setText(jsonConverter.toJsonString(request.getBody()))
                            .build()
            );
        }

        // Add parameters to request
        request.getQueryParams().forEach(item -> builder.addParameter(item.getLeft(), item.getRight()));

        // Add headers to request
        getAisRequiredHeaders(request).forEach(item -> builder.setHeader(item.getLeft(), item.getRight()));
        request.getHeaders().forEach(item -> builder.setHeader(item.getLeft(), item.getRight()));

        // Build the HttpUriRequest object
        return builder.build();
    }

    protected List<Pair<String, String>> getAisRequiredHeaders(AisHttpRequest request) {
        List<Pair<String, String>> headers = new ArrayList<>();
        headers.add(new Pair<>(BAPI_CORRELATION_ID_KEY, request.getCorrelationId()));
        headers.add(new Pair<>(BAPI_CHANNEL_ID_KEY, properties.getBapiChannelId()));
        headers.add(new Pair<>(BAPI_APP_ID_KEY, properties.getBapiAppId()));
        headers.add(new Pair<>(BAPI_REQUEST_ID_KEY, request.getRequestId()));
        headers.add(new Pair<>(BAPI_REQUEST_TIMESTAMP_KEY, LocalDateTime.now(ZoneOffset.UTC).format(BAPI_REQ_TIMESTAMP_FORMAT)));
        return headers;
    }

    /**
     * Return a URI by concatenating the base URL and the path.<br>
     * for example:
     * <pre>baseUrl = "https://api.example.com"</pre>
     * <pre>path = "/v1/resource/param1/value1/param2/value2"</pre>
     * <pre>the result will be "https://api.example.com/v1/resource/param1/value1/param2/value2"</pre>
     *
     * @param baseUrl base URL
     * @param path path
     * @return a URI string
     */
    private String getUri(String baseUrl, String path) {
        if (StringUtils.isBlank(baseUrl)) {
            throw new IllegalArgumentException("baseUrl cannot be null or empty.");
        }

        try {
            // encode special characters in the path
            String encodedPath = new URIBuilder().setPath(path).build().toString();
            return StringUtils.removeEnd(baseUrl, "/").concat("/")
                     .concat(StringUtils.removeStart(encodedPath, "/"));
        } catch (URISyntaxException e) {
            // rethrow the exception as an RuntimeException so caller do not need to handle it explicitly
            throw new IllegalArgumentException("path is not a valid URI.", e);
        }
    }
}
