package uk.ac.warwick.util.ais.core.apache;

import com.fasterxml.jackson.databind.JsonNode;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.entity.ContentType;
import org.apache.http.util.EntityUtils;
import uk.ac.warwick.util.ais.core.exception.AisHttpException;
import uk.ac.warwick.util.ais.core.httpclient.AbstractAisHttpResponseHandler;
import uk.ac.warwick.util.ais.core.json.AisJsonConverter;

import java.io.IOException;

/**
 * The default StuTalk API response handler to deserialise received JSON string into Java objects.
 */
public class DefaultAisHttpResponseHandler extends AbstractAisHttpResponseHandler<HttpResponse> {

    private static final String EMPTY_ARRAY_JSON = "[]";

    public DefaultAisHttpResponseHandler(AisJsonConverter jsonConverter) {
        super(jsonConverter);
    }

    @Override
    protected JsonNode handleResponseEntity(HttpResponse response) throws IOException {

        final StatusLine statusLine = response.getStatusLine();
        final HttpEntity entity = response.getEntity();
        if (statusLine.getStatusCode() >= 300) {
            // ensures the content body is always consumed
            String responseBody = entity == null ? null : EntityUtils.toString(entity);
            throw new AisHttpException(
                    AisHttpException.ErrorType.HTTP_STATUS_CODE_ERROR,
                    statusLine.getReasonPhrase(),
                    statusLine.getStatusCode(),
                    responseBody
            );
        }

        // There are some cases where we may receive empty/null JSON content, so we can treat it as an empty array
        // An empty array is also considered as a null object in case the expected class is an object class.
        if (entity == null) return jsonConverter.toJsonNode(EMPTY_ARRAY_JSON);

        String mimeType = ContentType.get(entity).getMimeType();
        String bodyString = EntityUtils.toString(entity); // Consume the entity to release the connection back to the pool
        return handleEntity(bodyString, mimeType);
    }
}
