package uk.ac.warwick.util.ais.apim.stutalk.json;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import uk.ac.warwick.util.ais.core.exception.AisJsonProcessingException;
import uk.ac.warwick.util.ais.core.json.AbstractAisJsonConverter;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * JSON response from StuTalk API always present in the format below,
 * the data object we expect to work with is wrapped in some layers that don't really make any sense.
 * The purpose of this class is to wrap/unwrap those additional layers
 * so that our data model classes can be defined in a simple structure and more meaningful way.
 *
 * <pre>
 * {
 *     "EXCHANGE": {
 *       "SRA": {
 *         "SRA.CAMS": [
 *           {
 *             "AYR_CODE.SRA.CAMS": "24/25",
 *             "MAV_OCCUR.SRA.CAMS": "A",
 *             "MOD_CODE.SRA.CAMS": "EN107-30",
 *             "PSL_CODE.SRA.CAMS": "Y",
 *             "SPR_CODE.SRA.CAMS": "5514251/1",
 *             "SRA_RSEQ.SRA.CAMS": "001",
 *             "AST_CODE.SRA.CAMS": "REP",
 *           }
 *         ]
 *       }
 *     }
 * }
 * </pre>
 */
public class StuTalkJsonConverterImpl extends AbstractAisJsonConverter implements StuTalkJsonConverter {

    public StuTalkJsonConverterImpl(ObjectMapper objectMapper) {
        super(objectMapper);
    }

    @Override
    public <T> T fromJsonNode(JsonNode json, TypeReference<T> typeReference) {
        if (json == null) throw new IllegalArgumentException("jsonNode must not be null.");
        List<String> properties = getStuTalkProperties(typeReference);
        return super.fromJsonNode(unwrap(json, properties), typeReference);
    }

    @Override
    public <T> T fromJsonNode(JsonNode json, Class<T> clazz) {
        if (json == null) throw new IllegalArgumentException("jsonNode must not be null.");
        List<String> properties = getStuTalkProperties(clazz);
        return super.fromJsonNode(unwrap(json, properties), clazz);
    }

    @Override
    public <T> T fromJsonString(String json, TypeReference<T> typeReference) {
        if (json == null) throw new IllegalArgumentException("json string must not be null.");
        List<String> properties = getStuTalkProperties(typeReference);
        JsonNode unwrappedNodes = unwrap(toJsonNode(json), properties);
        return super.fromJsonNode(unwrappedNodes, typeReference);
    }

    @Override
    public <T> T fromJsonString(String json, Class<T> clazz) {
        if (json == null) throw new IllegalArgumentException("json string must not be null.");
        List<String> properties = getStuTalkProperties(clazz);
        JsonNode unwrappedNodes = unwrap(toJsonNode(json), properties);
        return super.fromJsonNode(unwrappedNodes, clazz);
    }

    @Override
    public String toJsonString(Object obj) {
        if (obj == null) return  "{}"; // Return empty object instead of null.
        List<String> properties = getStuTalkProperties(obj);
        return super.toJsonString(wrap(objectMapper.valueToTree(obj), Lists.reverse(properties)));
    }

    private JsonNode wrap(JsonNode jsonNode, List<String> properties) {
        JsonNode result = jsonNode;
        for (String property : properties) {
            result = objectMapper.createObjectNode().set(property, result);
        }
        return result;
    }

    private JsonNode unwrap(JsonNode jsonNode, List<String> properties) {
        JsonNode result = jsonNode;
        for (String property: properties) {
            try {
                result = Objects.nonNull(result.get(property)) ? result.get(property) : result;
            } catch (Exception e) {
                throw new AisJsonProcessingException("Unwrap json failed.", e);
            }
        }
        return result;
    }

    /**
     * Get StuTalk properties of the object
     * If value is null (or has no StuTalkProperties) it will return an empty Array.
     */
    private List<String> getStuTalkProperties(Object value) {
        if (value == null) return Collections.emptyList();

        Class<?> clazz = getClassType(value);
        StuTalkProperties annotation = clazz.getAnnotation(StuTalkProperties.class);
        return (annotation != null) ? Arrays.asList(annotation.values()) : Collections.emptyList();
    }

    private Class<?> getClassType(Object value) {
        if (value instanceof Iterable) {
            return Iterables.get((Iterable<?>) value, 0).getClass();
        } else if (value instanceof scala.collection.Iterable) {
            return ((scala.collection.Iterable<?>) value).head().getClass();
        } else if (value instanceof TypeReference) {
            TypeReference<?> typeReference = (TypeReference<?>) value;
            if (typeReference.getType() instanceof ParameterizedType) {
                Type type = ((ParameterizedType) typeReference.getType()).getActualTypeArguments()[0];
               return (Class<?>) type;
            } else {
                return (Class<?>) typeReference.getType();
            }
        } else if (value instanceof Class) {
            return (Class<?>) value;
        } else {
            return value.getClass();
        }
    }
}
